/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2023
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.loadable.core

import groovy.transform.CompileStatic
import org.gradle.api.Project
import org.gradle.api.artifacts.Configuration
import org.gradle.api.artifacts.ConfigurationContainer
import org.gradle.api.model.ObjectFactory
import org.gradle.api.provider.Property
import org.gradle.api.provider.Provider
import org.ysb33r.grolifant.api.core.ProjectOperations
import org.ysb33r.grolifant.api.core.ProjectTools
import org.ysb33r.grolifant.api.core.StringTools
import org.ysb33r.grolifant.internal.core.IterableUtils
import org.ysb33r.grolifant.internal.core.Transform

import java.util.function.Function

import static org.ysb33r.grolifant.internal.core.IterableUtils.treatAsIterable

/**
 *
 * @author Schalk W. Cronje
 *
 * @since 2.0
 */
@SuppressWarnings('AbstractClassWithoutAbstractMethod')
@CompileStatic
abstract class ProjectToolsProxy implements ProjectTools {
    /**
     * Resolves an arbitrary item to a {@link Configuration} instance.
     *
     * @param configurationThingy Instance or {@link Configuration} or something that resolves to a string.
     * @return Configuration
     */
    @Override
    Configuration asConfiguration(Object configurationThingy) {
        Transform.convertItem(
            configurationThingy,
            transformer
        )
    }

    /**
     * Resolves arbitrary items to a collection of {@link Configuration} instances.
     *
     * @param configurationThingies Collection that might contain {@link Configuration} or string-type instances
     * @return Collection of resolved {@link Configuration} instances
     */
    @Override
    Collection<Configuration> asConfigurations(Collection<?> configurationThingies) {
        Set<Configuration> resolvedConfigurations = []
        configurationThingies.forEach { thingy ->
            switch (thingy) {
                case Configuration:
                    resolvedConfigurations.add((Configuration) thingy)
                    break
                case Collection:
                    resolvedConfigurations.addAll(asConfigurations((Collection) thingy))
                    break
                default:
                    if (treatAsIterable(thingy)) {
                        final asList = IterableUtils.toList((Iterable) thingy)
                        resolvedConfigurations.addAll(asConfigurations((Collection) asList))
                    } else {
                        resolvedConfigurations.add(asConfiguration(thingy))
                    }
            }
        }
        resolvedConfigurations
    }

    /**
     * Lazy-evaluated project version.
     *
     * @return Provider to project version
     */
    @Override
    Provider<String> getVersionProvider() {
        this.versionProvider
    }

    /**
     * Sets a new version for a project.
     * <p>This creates an internal objhect that can safely be evaluated by
     * Gradle's {@link org.gradle.api.Project#getVersion}.
     * </p>
     *
     * @param newVersion Anything that can be converted to a string using
     * {@link org.ysb33r.grolifant.api.core.StringTools}.
     */
    @Override
    void setVersionProvider(Object newVersion) {
        gradleProjectVersion.version = newVersion
    }

    protected ProjectToolsProxy(ProjectOperations po, Project tempProjectReference) {
        this.projectOperations = po
        this.configurations = tempProjectReference.configurations
        this.gradleProjectVersion = new GradleProjectVersion(this, tempProjectReference.objects)

        final initialVersion = tempProjectReference.version?.toString() ?: unspecifiedProjectVersion
        this.gradleProjectVersion.updateVersion(tempProjectReference.provider { -> initialVersion })

        this.versionProvider = tempProjectReference.provider { ->
            if (tempProjectReference.version instanceof GradleProjectVersion) {
                gradleProjectVersion.toString()
            } else {
                po.stringTools.stringize(tempProjectReference.version)
            }
        }

        this.legacyVersionUpdater = { ->
            tempProjectReference.version = gradleProjectVersion
        }

        updateLegacyVersion()
    }

    protected final String unspecifiedProjectVersion = 'unspecified'
    protected final String unspecifiedProjectGroup = ''
    protected final ProjectOperations projectOperations

    protected void updateVersion(Provider<String> newVersion) {
        this.gradleProjectVersion.updateVersion(newVersion)
        updateLegacyVersion()
    }

    protected void updateLegacyVersion() {
        legacyVersionUpdater.call()
    }

    protected static class GradleProjectVersion {
        GradleProjectVersion(ProjectToolsProxy owner, ObjectFactory objectFactory) {
            this.owner = owner
            this.version = objectFactory.property(String)
        }

        void updateVersion(Provider<String> ver) {
            version.set(ver)
        }

        void setVersion(Object ver) {
            owner.projectOperations.stringTools.updateStringProperty(version, ver)
        }

        @Override
        String toString() {
            version.get()
        }

        private final ProjectToolsProxy owner
        private final Property<String> version
    }

    private Function<Object, Configuration> getTransformer() {
        CONFIGURATION_TRANSFORMER.curry(
            configurations,
            projectOperations.stringTools
        ) as Function<Object, Configuration>
    }

    private final ConfigurationContainer configurations
    private final GradleProjectVersion gradleProjectVersion
    private final Provider<String> versionProvider
    private final Closure legacyVersionUpdater

    private static final Closure CONFIGURATION_TRANSFORMER = {
        ConfigurationContainer c, StringTools st, Object configurationThingy ->
            switch (configurationThingy) {
                case Configuration:
                    (Configuration) configurationThingy
                    break
                default:
                    c.getByName(st.stringize(configurationThingy))
            }
    }

}
