/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2023
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.api.core.runnable;

import groovy.lang.Closure;
import groovy.lang.DelegatesTo;
import org.gradle.api.Action;
import org.gradle.api.tasks.Internal;
import org.gradle.api.tasks.Nested;
import org.gradle.process.ExecSpec;
import org.ysb33r.grolifant.api.core.CmdlineArgumentSpec;
import org.ysb33r.grolifant.api.core.ProjectOperations;
import org.ysb33r.grolifant.api.core.executable.AppRunnerSpec;
import org.ysb33r.grolifant.api.core.executable.CmdLineArgumentSpecEntry;
import org.ysb33r.grolifant.api.core.executable.ExecutableEntryPoint;
import org.ysb33r.grolifant.api.core.executable.ProcessExecutionSpec;

import java.util.Collection;

/**
 * Base class for creating execution specifications.
 *
 * @param <T> CLass extending {@link AbstractExecSpec}.
 * @author Schalk W. Cronjé
 * @since 2.0
 */
public class AbstractExecSpec<T extends AbstractExecSpec<T>> implements Executable<T> {

    /**
     * Configures the native entrypoint.
     *
     * @param configurator An action to configure the external executable's entrypoint.
     */
    @Override
    public void entrypoint(Action<ExecutableEntryPoint> configurator) {
        appRunnerSpec.configureEntrypoint(configurator);
    }

    /**
     * Configures the native entrypoint.
     *
     * @param configurator A closure to configure the external executable's entrypoint.
     */
    @Override
    public void entrypoint(@DelegatesTo(ExecutableEntryPoint.class) Closure<?> configurator) {
        appRunnerSpec.configureEntrypoint(configurator);
    }

    /**
     * Configures the arguments.
     *
     * @param configurator An action to configure the arguments.
     */
    @Override
    public void runnerSpec(Action<CmdlineArgumentSpec> configurator) {
        appRunnerSpec.configureCmdline(configurator);
    }

    /**
     * Configures the arguments.
     *
     * @param configurator A closure to configure the arguments.
     */
    @Override
    public void runnerSpec(@DelegatesTo(CmdlineArgumentSpec.class) Closure<?> configurator) {
        appRunnerSpec.configureCmdline(configurator);
    }

    /**
     * Configures the stream redirections and exit checks.
     *
     * @param configurator An action to configure the execution.
     */
    @Override
    public void process(Action<ProcessExecutionSpec> configurator) {
        appRunnerSpec.configureProcess(configurator);
    }

    /**
     * Configures the stream redirections and exit checks.
     *
     * @param configurator A closure to configure the execution.
     */
    @Override
    public void process(@DelegatesTo(ProcessExecutionSpec.class) Closure<?> configurator) {
        appRunnerSpec.configureProcess(configurator);
    }

    /**
     * Copies this specification to a standard {@link ExecSpec}.
     *
     * @param spec Target execution specification.
     */
    public void copyTo(ExecSpec spec) {
        appRunnerSpec.copyTo(spec);
    }

    protected AbstractExecSpec(ProjectOperations po) {
        this.appRunnerSpec = po.getExecTools().appRunnerSpec();
        this.projectOperations = po;
    }

    /**
     * Adds a command-line processor that will process command-line arguments in a specific order.
     * <p>
     * For instance in a script, one want to process the exe args before the script args.
     * <p>
     * In a system that has commands and subcommands, one wants to process this in the order of exe args, command args,
     * and then subcommand args.
     * <p>
     * This method allows the implementor to control the order of processing  for all the groupings.
     *
     * @param name      Name of command-line processor.
     * @param order     Order in queue to process.
     * @param processor The specific grouping.
     */
    protected void addCommandLineProcessor(String name, Integer order, CmdlineArgumentSpec processor) {
        appRunnerSpec.addCommandLineProcessor(name, order, processor);
    }

    @Internal
    protected Collection<CmdLineArgumentSpecEntry> getCommandLineProcessors() {
        return appRunnerSpec.getCommandLineProcessors();
    }

    @Internal
    protected final ProjectOperations projectOperations;

    @Nested
    protected final AppRunnerSpec appRunnerSpec;
}
