/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2024
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.loadable.core

import groovy.transform.CompileStatic
import org.gradle.StartParameter
import org.gradle.api.Action
import org.gradle.api.Project
import org.gradle.api.file.CopySpec
import org.gradle.api.file.DeleteSpec
import org.gradle.api.file.FileCollection
import org.gradle.api.file.FileTree
import org.gradle.api.invocation.Gradle
import org.gradle.api.logging.LogLevel
import org.gradle.api.logging.configuration.ConsoleOutput
import org.gradle.api.provider.Property
import org.gradle.api.provider.Provider
import org.gradle.api.provider.ProviderFactory
import org.gradle.api.resources.ReadableResource
import org.gradle.api.resources.ResourceHandler
import org.gradle.api.tasks.WorkResult
import org.gradle.process.ExecResult
import org.gradle.process.ExecSpec
import org.gradle.process.JavaExecSpec
import org.ysb33r.grolifant.api.core.ArchiveOperationsProxy
import org.ysb33r.grolifant.api.core.ConfigurationTools
import org.ysb33r.grolifant.api.core.ExecOperationsProxy
import org.ysb33r.grolifant.api.core.ExecTools
import org.ysb33r.grolifant.api.core.FileSystemOperations
import org.ysb33r.grolifant.api.core.GradleSysEnvProvider
import org.ysb33r.grolifant.api.core.JvmTools
import org.ysb33r.grolifant.api.core.ProjectOperations
import org.ysb33r.grolifant.api.core.ProjectTools
import org.ysb33r.grolifant.api.core.ProviderTools
import org.ysb33r.grolifant.api.core.RepositoryTools
import org.ysb33r.grolifant.api.core.StringTools
import org.ysb33r.grolifant.api.core.TaskTools
import org.ysb33r.grolifant.api.errors.NotSupportedException
import org.ysb33r.grolifant.internal.core.ConfigurationCache

@SuppressWarnings('MethodCount')
@CompileStatic
abstract class ProjectOperationsProxy implements ProjectOperations {
    /**
     * Creates resource that points to a bzip2 compressed file at the given path.
     *
     * @param compressedFile File evaluated as per {@link #file}.
     * @return Readable resource
     */
    @Override
    ReadableResource bzip2Resource(Object compressedFile) {
        this.resourceHandler.bzip2(file(compressedFile))
    }

    /**
     * Creates resource that points to a gzip compressed file at the given path.
     * @param compressedFile File evaluated as per {@link #file}.
     * @return Readable resource
     */
    @Override
    ReadableResource gzipResource(Object compressedFile) {
        this.resourceHandler.gzip(file(compressedFile))
    }

    /**
     * Performs a copy according to copy specification.
     *
     * @param action Copy specification
     * @return Result of copy operation.
     */
    @Override
    WorkResult copy(Action<? super CopySpec> action) {
        fsOperations.copy(action)
    }

    @Override
    boolean delete(Object... paths) {
        delete { DeleteSpec ds ->
            ds.delete(fileize(paths as List))
        }.didWork
    }

    @Override
    WorkResult delete(Action<? super DeleteSpec> action) {
        fsOperations.delete(action)
    }

    /**
     * Executes the specified external process.
     *
     * @param action
     * @return {@link ExecResult} that can be used to check if the execution worked.
     *
     */
    @Override
    ExecResult exec(Action<? super ExecSpec> action) {
        execOperations.exec(action)
    }

    /** Convert an object to a file
     *
     * @param path Path to convert.
     * @return File object
     */
    @Override
    File file(Object path) {
        fsOperations.file(path)
    }

    /**
     * Similar to {@Link #file}, but does not throw an exception if the object is {@code null} or an empty provider.
     *
     * @param file Potential {@link File} object
     * @return File instance or {@code null}.
     *
     * @since 1.2
     */
    @Override
    File fileOrNull(Object file) {
        fsOperations.fileOrNull(file)
    }

    /**
     * Converts a collection of file-like objects to a list of  {@link java.io.File} instances with project context.
     * <p>
     * It will convert anything that the singular version of {@code FileUtils.fileize} can do.
     * In addition it will recursively resolve any collections that result out of resolving the supplied items.
     *
     * @param files List of object to evaluate
     * @return List of resolved files.
     */
    @Override
    List<File> fileize(Iterable<Object> files) {
        fsOperations.files(files.toList()).files.toList()
    }

    /**
     * <p>Creates a {@link FileCollection} containing the given files, as defined by {@link Project#files(Object ...)}.
     *
     * <p>This method can also be used to create an empty collection, but the collection may not be mutated later.</p>
     *
     * @param paths The paths to the files. May be empty.
     * @return The file collection. Never returns null.
     *
     * @deprecated USe {@link FileSystemOperations#files} instead.
     */
    @Override
    FileCollection files(Object... paths) {
        fsOperations.files(paths.toList())
    }

    /**
     * Tools to deal with Gradle configurations.
     * @return
     */
    @Override
    ConfigurationTools getConfigurations() {
        this.configurationTools
    }

    /**
     * Tools to deal with out-of-process, non-JVM, executables.
     *
     * @return Tools instance optimised for current version of Gradle (where possible).
     */
    @Override
    ExecTools getExecTools() {
        this.execTools
    }

    /**
     * Returns an object instance for filesystem operations that deals coprrectly with the functionality of the
     * curretn Gradle version.
     *
     * @return Instance of {@link FileSystemOperations}
     */
    @Override
    FileSystemOperations getFsOperations() {
        this.fsOperations
    }

    /**
     * Gradle distribution home directory.
     *
     * @return Directory.
     *
     * @since 2.2
     */
    @Override
    Provider<File> getGradleHomeDir() {
        this.gradleHomeDir
    }

    /**
     * Gradle user home directory. Usually {@code ~/.gradle} on non -Windows.
     *
     * @return Directory.
     */
    @Override
    Provider<File> getGradleUserHomeDir() {
        this.gradleUserHomeDir
    }

    /**
     * A reference to the provider factory.
     *
     * @return {@link ProviderFactory}
     */
    @Override
    ProviderFactory getProviders() {
        this.providerFactory
    }

    /**
     * Utilities for working with tasks in a consistent manner across Gradle versions.
     *
     * @return {@link TaskTools} instance.
     *
     * @since 1.3
     */
    @Override
    TaskTools getTasks() {
        this.taskTools
    }

    /**
     * Whether configuration cache is enabled for a build.
     *
     * @return {@code true} is configuration cache is available and enabled.
     */
    @Override
    boolean isConfigurationCacheEnabled() {
        this.configurationCacheEnabled
    }

    /**
     * Whether Gradle is operating in offline mode.
     *
     * @return {@code true} if offline.
     */
    @Override
    boolean isOffline() {
        this.offline
    }

    /**
     * Whether dependencies should be refreshed.
     *
     * @return {@code true} to check dependencies again.
     */
    @Override
    boolean isRefreshDependencies() {
        this.refreshDependencies
    }

    /**
     * Whether tasks should be re-ruin
     *
     * @return {@code true} if tasks were set to be re-run.
     */
    @Override
    boolean isRerunTasks() {
        this.rerunTasks
    }

    /**
     * Executes the specified external java process.
     * @param action
     * @return {@link ExecResult} that can be used to check if the execution worked.
     *
     */
    @Override
    ExecResult javaexec(Action<? super JavaExecSpec> action) {
        execOperations.javaexec(action)
    }

    /**
     * Get the minimum log level for Gradle.
     *
     * @return Log level
     */
    @Override
    LogLevel getGradleLogLevel() {
        this.logLevel
    }

    /**
     * Console output mode
     *
     * @return How the console output has been requested.
     */
    @Override
    ConsoleOutput getConsoleOutput() {
        this.consoleOutput
    }

    /**
     * Get the full project path including the root project name in case of a multi-project.
     *
     * @return The fully qualified project path including root project.
     *
     * @since 1.2
     */
    @Override
    String getFullProjectPath() {
        this.fullProjectPath
    }

    /**
     * Lazy-evaluated project group.https://gitlab.com/ysb33rOrg/grolifant/-/jobs/3248448094
     *
     * @return provider to project group
     */
    @Override
    Provider<String> getGroupProvider() {
        projectTools.groupProvider
    }

    /**
     * Tools for working with JVMs
     *
     * @return The appropriate provider for the Gradle version.
     *
     * @since 2.0
     */
    JvmTools getJvmTools() {
        this.jvmTools
    }

    /**
     * Returns the project directory.
     *
     * @return Project directory.
     */
    @Override
    File getProjectDir() {
        this.projectDir
    }

    /**
     * The project name
     *
     * @return Cached value of project name.
     */
    @Override
    String getProjectName() {
        this.projectName
    }

    /**
     * Get project path.
     *
     * @return The fully qualified project path.
     *
     * @since 1.2
     */
    @Override
    String getProjectPath() {
        this.projectPath
    }

    /**
     * Returns the root directory of the project.
     *
     * @return Root directory.
     *
     * @since 2.0
     */
    @Override
    File getProjectRootDir() {
        this.rootDir
    }

    /**
     * Tools to deal with project & configuration specifics down to Gradle 4.0.
     *
     * @return Tools instance optimised for current version of Gradle (where possible).
     */
    @Override
    ProjectTools getProjectTools() {
        this.projectTools
    }

    /**
     * Tools to deal with provider down to Gradle 4.0.
     *
     * @return Tools instance optimised for current version of Gradle (where possible).
     */
    @Override
    ProviderTools getProviderTools() {
        this.providerTools
    }

    /**
     * Tools for dealing with repositories.
     *
     * @return Tools instance.
     *
     * @since 2.0
     */
    @Override
    RepositoryTools getRepositoryTools() {
        this.repositoryTools
    }

    /**
     * Tools for dealing with conversions of various objects into string or lists of strings.
     *
     * @return String tools instance.
     *
     * @since 2.0
     */
    @Override
    StringTools getStringTools() {
        this.stringTools
    }

    /**
     * Lazy-evaluated project version.
     *
     * @return Provider to project version
     */
    @Override
    Provider<String> getVersionProvider() {
        projectTools.versionProvider
    }

    /**
     * Creates a provider to an environmental variable.
     *
     * @param name Anything convertible to a string
     * @param configurationTimeSafety Whether this property can be read safely at configuration time. It is suggested
     *                                to just use {@link #atConfigurationTime} for this parameter
     * @return Provider to the value of the an environmental variable.
     */
    @Override
    Provider<String> environmentVariable(Object name, boolean configurationTimeSafety) {
        propertyProvider.environmentVariable(name, configurationTimeSafety)
    }

    /**
     * Creates a provider to a project property.
     *
     * @param name Anything convertible to a string
     * @param configurationTimeSafety Whether this property can be read safely at configuration time. It is suggested
     *                                to just use {@link #atConfigurationTime} for this parameter
     * @return Provider to the value of the project property.
     */
    @Override
    Provider<String> gradleProperty(Object name, boolean configurationTimeSafety) {
        propertyProvider.gradleProperty(name, configurationTimeSafety)
    }

    /**
     * Whether current project is the root project.
     *
     * @return {@code true} is current project is root project.
     *
     * @since 1.2
     */
    @Override
    boolean isRoot() {
        this.root
    }

    /**
     * Returns the relative path from the project directory to the given path.
     *
     * @param f Object that is resolvable to a file within project context
     *
     * @return Relative path. Never {@code null}.
     */
    @Override
    String relativePath(Object f) {
        fsOperations.relativePath(f)
    }

    /**
     *
     * Searches by Gradle property, then system property and finally by environment variable using the
     * {@code PropertyResolver convention}.
     *
     * @param name Anything convertible to a string
     * @param defaultValue Default value to return if the property search order does not return any value.
     *                                Can be {@code null}. Anything convertible to a string.
     * @param configurationTimeSafety Whether this property can be read safely at configuration time. It is suggested
     *                                to just use {@link #atConfigurationTime} for this parameter.
     * @return Provider to finding a property by the specified name.
     */
    @Override
    Provider<String> resolveProperty(Object name, Object defaultValue, boolean configurationTimeSafety) {
        propertyProvider.resolve(name, defaultValue, configurationTimeSafety)
    }

    /**
     * Creates a provider to a system property.
     *
     * @param name Anything convertible to a string
     * @param configurationTimeSafety Whether this property can be read safely at configuration time. It is suggested
     *                                to just use {@link #atConfigurationTime} for this parameter
     * @return Provider to the value of the system property.
     */
    @Override
    Provider<String> systemProperty(Object name, boolean configurationTimeSafety) {
        propertyProvider.resolve(name, null, configurationTimeSafety)
    }

    /**
     * Returns a TAR tree presentation
     *
     * @param tarPath Path to tar file
     * @return File tree
     */
    @Override
    FileTree tarTree(Object tarPath) {
        archiveOperations.tarTree(tarPath)
    }

    /**
     * Updates a file provider.
     * <p>
     * Update property or otherwise the provider will be assigned a new Provider instance.
     *
     * @param provider Current property
     * @param file Value that should be lazy-resolved to a file using {@link #file}.
     */
    @Override
    void updateFileProperty(Provider<File> provider, Object file) {
        if (provider instanceof Property) {
            fsOperations.updateFileProperty((Property<File>) provider, file)
        } else {
            throw new NotSupportedException('Provider has to be Property<File>')
        }
    }

    /**
     * Updates a string provider.
     * <p>
     * Update property or otherwise the provider will be assigned a new Provider instance.
     *
     * @param provider Current property
     * @param str Value that should be lazy-resolved to a string .
     */
    @Override
    void updateStringProperty(Provider<String> provider, Object str) {
        if (provider instanceof Property) {
            stringTools.updateStringProperty((Property<String>) provider, str)
        } else {
            throw new NotSupportedException('Provider has to be Property<String>')
        }
    }

    /**
     * Returns a ZIP tree presentation
     *
     * @param zipPath Path to zip file
     * @return File tree
     */
    @Override
    FileTree zipTree(Object zipPath) {
        archiveOperations.zipTree(zipPath)
    }

    abstract protected ArchiveOperationsProxy getArchiveOperations()

    abstract protected ExecOperationsProxy getExecOperations()

    abstract protected GradleSysEnvProvider getPropertyProvider()

    protected ProjectOperationsProxy(Project tempProjectReference) {
        this.providerFactory = tempProjectReference.providers
        this.resourceHandler = tempProjectReference.resources
        this.fsOperations = FileSystemOperations.load(this, tempProjectReference)
        this.taskTools = TaskTools.load(this, tempProjectReference)
        this.repositoryTools = RepositoryTools.load(this, tempProjectReference)
        this.providerTools = ProviderTools.load(tempProjectReference)
        this.projectTools = ProjectTools.load(this, tempProjectReference)
        this.stringTools = StringTools.load(tempProjectReference)
        this.jvmTools = JvmTools.load(this, tempProjectReference)
        this.execTools = ExecTools.load(this, tempProjectReference)
        this.configurationTools = ConfigurationTools.load(this, tempProjectReference)

        Gradle gradle = tempProjectReference.gradle
        StartParameter startParameter = gradle.startParameter

        this.root = tempProjectReference == tempProjectReference.rootProject
        this.offline = startParameter.offline
        this.logLevel = startParameter.logLevel
        this.refreshDependencies = startParameter.refreshDependencies
        this.rerunTasks = startParameter.rerunTasks
        this.gradleHomeDir = providerFactory.provider({ File f -> f }.curry(gradle.gradleHomeDir))
        this.gradleUserHomeDir = providerFactory.provider({ File f -> f }.curry(gradle.gradleUserHomeDir))
        this.consoleOutput = startParameter.consoleOutput
        this.projectDir = tempProjectReference.projectDir
        this.rootDir = tempProjectReference.rootDir
        this.configurationCacheEnabled = ConfigurationCache.isEnabled(tempProjectReference)
        this.projectName = tempProjectReference.name
        this.projectPath = tempProjectReference.path
        this.fullProjectPath = root ? this.projectName : "${tempProjectReference.rootProject.name}${projectPath}"
    }

    private final ProviderFactory providerFactory
    private final Provider<File> gradleUserHomeDir
    private final Provider<File> gradleHomeDir
    private final File projectDir
    private final File rootDir
    private final String projectName
    private final String projectPath
    private final String fullProjectPath
    private final boolean root
    private final boolean offline
    private final boolean refreshDependencies
    private final boolean rerunTasks
    private final LogLevel logLevel
    private final ConsoleOutput consoleOutput
    private final ResourceHandler resourceHandler
    private final boolean configurationCacheEnabled
    private final FileSystemOperations fsOperations
    private final TaskTools taskTools
    private final RepositoryTools repositoryTools
    private final ProviderTools providerTools
    private final ProjectTools projectTools
    private final StringTools stringTools
    private final JvmTools jvmTools
    private final ExecTools execTools
    private final ConfigurationTools configurationTools
}
