/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2024
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.api.core.executable;

import java.io.InputStream;
import java.io.OutputStream;

/**
 * Describes the data stream redirections for a process including exit checks.
 * <p>
 * Similar to {@link org.gradle.process.BaseExecSpec} except to {@code getCommandLine / setCommandLine}.
 *
 * @author Schalk W. Cronjé
 * @since 2.0
 */
public interface ProcessExecutionSpec {

    /**
     * Tells whether a non-zero exit value is ignored, or an exception thrown.
     *
     * @return whether a non-zero exit value is ignored, or an exception thrown.
     */
    boolean isIgnoreExitValue();

    /**
     * Sets whether a non-zero exit value is ignored, or an exception thrown.
     *
     * @param ignoreExitValue - whether a non-zero exit value is ignored, or an exception thrown
     */
    void setIgnoreExitValue(boolean ignoreExitValue);

    /**
     * Returns the standard input stream for the process executing the command. The stream is closed after the
     * process completes.
     *
     * @return Input stream.
     */
    InputStream getStandardInput();

    /**
     * Returns the output stream to consume standard output from the process executing the command.
     *
     * @return The output stream.
     */
    OutputStream getStandardOutput();

    /**
     * Returns the output stream to consume standard error from the process executing the command.
     *
     * @return Error output stream
     */
    OutputStream getErrorOutput();

    /**
     * Sets the output stream to consume standard error from the process executing the command.
     *
     * @param outputStream The output stream.
     */
    void setErrorOutput(OutputStream outputStream);

    /**
     * Sets the standard input stream for the process executing the command.
     *
     * @param inputStream The standard input stream for the process. Must not be null.
     */
    void setStandardInput(InputStream inputStream);

    /**
     * Sets the output stream to consume standard output from the process executing the command.
     *
     * @param outputStream The standard output stream for the process. Must not be null.
     */
    void setStandardOutput(OutputStream outputStream);
}
