/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2024
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.loadable.core

import groovy.transform.CompileStatic
import org.gradle.api.provider.Property
import org.gradle.api.provider.ProviderFactory
import org.gradle.api.resources.TextResource
import org.ysb33r.grolifant.api.core.StringTools
import org.ysb33r.grolifant.api.errors.ChecksumCreationException
import org.ysb33r.grolifant.internal.core.Transform

import java.nio.file.Path
import java.security.MessageDigest
import java.security.NoSuchAlgorithmException
import java.util.function.Function

@CompileStatic
class StringToolsProxy implements StringTools {

    /**
     * Creates a SHA-256 has of a URI.
     *
     * @param uri URI to hash
     * @return SHA-257 hash that is hex-encoded.
     */
    @Override
    String hashUri(URI uri) {
        try {
            MessageDigest messageDigest = MessageDigest.getInstance('SHA256')
            messageDigest.update(uri.toString().bytes)
            new BigInteger(1, messageDigest.digest()).toString(36)
        } catch (NoSuchAlgorithmException e) {
            throw new ChecksumCreationException('Could not create SHA-256 checksum of URI', e)
        }
    }

    /**
     * Get final package or directory name from a URI
     *
     * @param uri URI to process
     * @return Last part of URI path.
     */
    @Override
    String packageNameFromUri(URI uri) {
        final String path = uri.path
        int p = path.lastIndexOf('/')
        (p < 0) ? path : path.substring(p + 1)
    }

    /**
     * Converts most things to a string. Closures are evaluated as well.
     *
     * @param stringy An object that can be converted to a string or a closure that
     *                can be evaluated to something that can be converted to a string.
     * @return A string object
     */
    @Override
    String stringize(Object stringy) {
        Transform.convertItem(stringy, STRING_TRANSFORMER)
    }

    /**
     * Like {@link #stringize}, but returns {@code null} rather than throwing an exception, when item is {@code null},
     * an empty {@Link Provider} or an empty {@link java.util.Optional}.
     *
     * @param stringy
     * @return string or {@code null}
     */
    @Override
    String stringizeOrNull(Object stringy) {
        Transform.convertItemOrNull(stringy, STRING_TRANSFORMER_NULL)
    }

    /**
     * Converts a collection of most things to a list of strings. Closures are evaluated as well.
     *
     * @param stringyThings Iterable list of objects that can be converted to strings.
     * @return A list of strings
     */
    @Override
    List<String> stringize(Collection<?> stringyThings) {
        List<String> output = []
        Transform.convertItems(stringyThings, output, STRING_TRANSFORMER)
        output
    }

    /**
     * Like {@link #stringize}, but drops any nulls, or empty instances of {@link Provider} and {@link Optional}.
     *
     * @param stringyThings
     * @return A list of strings
     */
    @Override
    List<String> stringizeDropNull(Collection<?> stringyThings) {
        List<String> output = []
        Transform.convertItemsDropNull(stringyThings, output, STRING_TRANSFORMER_NULL)
        output
    }

    /**
     * Create a URI where the user/password is masked out.
     *
     * @param uri Original URI
     * @return URI with no credentials.
     */
    @Override
    URI safeUri(URI uri) {
        new URI(uri.scheme, null, uri.host, uri.port, uri.path, uri.query, uri.fragment)
    }

    /**
     * Updates a {@code Property<String>}.
     *
     * THis provides a more powerful way than {@link Property#set}.
     *
     * @param provider String property
     * @param stringy Objexct that is a string or can be converted to a string.
     */
    @Override
    void updateStringProperty(Property<String> provider, Object stringy) {
        provider.set(providers.provider { -> stringizeOrNull(stringy) })
    }

    /**
     * Attempts to convert object to a URI.
     * <p>
     * Closures can be passed and will be evaluated. Result will then be converted to a URI.
     *
     * @param uriThingy Anything that could be converted to a URI
     * @return URI object
     */
    @Override
    URI urize(Object uriThingy) {
        Transform.convertItem(uriThingy, URI_TRANSFORMER.curry(this))
    }

    protected StringToolsProxy(ProviderFactory pf) {
        this.providers = pf
    }

    private final ProviderFactory providers

    private static final Closure URI_TRANSFORMER = { StringTools str, Object uri ->
        switch (uri) {
            case URI:
                return (URI) uri
            case URL:
                return ((URL) uri).toURI()
            case File:
                return ((File) uri).toURI()
            case Path:
                return ((Path) uri).toUri()
            default:
                str.stringize(uri).toURI()
        }
    }

    private static final Function<Object, String> STRING_TRANSFORMER = { str ->
        switch (str) {
            case TextResource:
                ((TextResource) str).asString()
                break
            default:
                str.toString()
        }
    } as Function<Object, String>

    private static final Function<Object, String> STRING_TRANSFORMER_NULL = { str ->
        str == null ? null : STRING_TRANSFORMER.apply(str)
    } as Function<Object, String>
}
