/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2022
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.internal.v4.jvm

import groovy.transform.CompileDynamic
import groovy.transform.CompileStatic
import groovy.transform.TypeChecked
import org.gradle.api.file.FileCollection
import org.gradle.api.provider.Provider
import org.gradle.process.JavaExecSpec
import org.ysb33r.grolifant.api.core.LegacyLevel
import org.ysb33r.grolifant.api.core.ProjectOperations
import org.ysb33r.grolifant.api.core.jvm.JvmEntryPoint
import org.ysb33r.grolifant.api.core.jvm.ModularitySpec
import org.ysb33r.grolifant.api.v4.StringUtils

import java.util.concurrent.Callable

@CompileStatic
class JvmEntryPointProxy implements JvmEntryPoint {

    JvmEntryPointProxy(
        JavaExecSpec es,
        ModularitySpec ms,
        ProjectOperations po
    ) {
        this.javaExecSpec = es
        this.modularitySpecProxy = ms
        this.projectOperations = po
    }

    @Delegate(interfaces = false, includes = ['getClasspath', 'setClasspath', 'classpath'])
    final JavaExecSpec javaExecSpec
    final ModularitySpec modularitySpecProxy

    JavaExecSpec setClasspath(FileCollection fc) {
        javaExecSpec.classpath = fc
        javaExecSpec
    }

    JavaExecSpec classpath(Object... paths) {
        javaExecSpec.classpath(paths)
        javaExecSpec
    }

    FileCollection getClasspath() { javaExecSpec.classpath }

    void setInferModulePath(Provider<Boolean> p) { modularitySpecProxy.inferModulePath = p }

    void setInferModulePath(boolean p) { modularitySpecProxy.inferModulePath = p }

    Provider<Boolean> getInferModulePath() { modularitySpecProxy.inferModulePath }

    void setMainModule(Object o) { modularitySpecProxy.mainModule = o }

    Provider<String> getMainModule() { modularitySpecProxy.mainModule }

    /**
     * Get the main class name.
     *
     * @return Provider.
     */
    @Override
    Provider<String> getMainClass() {
        getMainClassInternal(javaExecSpec)
    }

    @Override
    void setMainClass(Object name) {
        setMainClassInternal(javaExecSpec, name)
    }

    /**
     * Copy entry point information to a {@link JavaExecSpec}
     *
     * @param target {@link JavaExecSpec}
     */
    void copyTo(JavaExecSpec target) {
        setMainClassInternal(target, mainClass)
        target.classpath = classpath
    }

    @CompileDynamic
    @TypeChecked
    private Provider<String> mainNamePre64(JavaExecSpec jes) {
        projectOperations.provider({ JavaExecSpec es -> es.main }.curry(jes) as Callable<String>)
    }

    @CompileDynamic
    @TypeChecked
    private Provider<String> mainName(JavaExecSpec jes) {
        jes.mainClass
    }

    private Provider<String> getMainClassInternal(JavaExecSpec jes) {
        if (LegacyLevel.PRE_6_4) {
            mainNamePre64(jes)
        } else {
            mainName(jes)
        }
    }

    private void setMainClassInternal(JavaExecSpec jes, Object main) {
        if (LegacyLevel.PRE_6_4) {
            setMainNamePre64(jes, main)
        } else {
            setMainName(jes, main)
        }
    }

    @CompileDynamic
    @TypeChecked
    private void setMainNamePre64(JavaExecSpec jes, Object main) {
        jes.main = StringUtils.stringize(main)
    }

    @CompileDynamic
    @TypeChecked
    private void setMainName(JavaExecSpec jes, Object main) {
        projectOperations.updateStringProperty(jes.mainClass, main)
    }

    private final ProjectOperations projectOperations
}
