/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2022
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.loadable.v4

import groovy.transform.CompileStatic
import org.ysb33r.grolifant.api.core.StringTools
import org.ysb33r.grolifant.api.v4.StringUtils
import org.ysb33r.grolifant.api.v4.UriUtils

/**
 * Implements a set of string tools for Gradle 4.0+
 *
 * @since 2.0
 */
@CompileStatic
class DefaultStringTools implements StringTools {
    /**
     * Get final package or directory name from a URI
     *
     * @param uri URI to process
     * @return Last part of URI path.
     */
    @Override
    String packageNameFromUri(URI uri) {
       UriUtils.getPkgName(uri)
    }

    /**
     * Creates a SHA-256 has of a URI.
     *
     * @param uri URI to hash
     * @return SHA-257 hash that is hex-encoded.
     */
    @Override
    String hashURI(URI uri) {
        UriUtils.hashURI(uri)
    }

    /**
     * Create a URI where the user/password is masked out.
     *
     * @param uri Original URI
     * @return URI with no credentials.
     */
    @Override
    URI safeUri(URI uri) {
        UriUtils.safeUri(uri)
    }

    /**
     * Attempts to convert object to a URI.
     * <p>
     * Closures can be passed and will be evaluated. Result will then be converted to a URI.
     *
     * @param uriThingy Anything that could be converted to a URI
     * @return URI object
     */
    @Override
    URI urize(Object uriThingy) {
       UriUtils.urize(uriThingy)
    }

    /**
     * Converts most things to a string. Closures are evaluated as well.
     *
     * @param stringy An object that can be converted to a string or a closure that
     *                can be evaluated to something that can be converted to a string.
     * @return A string object
     */
    @Override
    String stringize(Object stringy) {
        StringUtils.stringize(stringy)
    }

    /**
     * Like {@link #stringize}, but returns {@code null} rather than throwing an exception, when item is {@code null},
     * an empty {@Link Provider} or an empty {@link java.util.Optional}.
     *
     * @param stringy
     * @return string or {@code null}
     */
    @Override
    String stringizeOrNull(Object stringy) {
        StringUtils.stringizeOrNull(stringy)
    }

    /**
     * Converts a collection of most things to a list of strings. Closures are evaluated as well.
     *
     * @param stringyThings Iterable list of objects that can be converted to strings, including closure
     *   that can be evaluated into objects that can be converted to strings.
     * @return A list of strings
     */
    @Override
    List<String> stringize(Iterable<?> stringyThings) {
        StringUtils.stringize(stringyThings)
    }

    /**
     * Like {@link #stringize}, but drops any nulls, or empty instances of {@code Provider} and {@link Optional}.
     *
     * @param stringyThings
     * @return A list of strings
     */
    @Override
    List<String> stringizeDropNull(Iterable<?> stringyThings) {
        StringUtils.stringizeDropNull(stringyThings)
    }
}
