/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2022
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.loadable.v4

import groovy.transform.CompileStatic
import groovy.util.logging.Slf4j
import org.gradle.api.Project
import org.gradle.api.Transformer
import org.gradle.api.file.ConfigurableFileTree
import org.gradle.api.file.CopySpec
import org.gradle.api.file.Directory
import org.gradle.api.internal.file.FileOperations
import org.gradle.api.internal.project.ProjectInternal
import org.gradle.api.provider.Provider
import org.gradle.api.provider.ProviderFactory
import org.ysb33r.grolifant.api.core.ArchiveOperationsProxy
import org.ysb33r.grolifant.api.core.ExecOperationsProxy
import org.ysb33r.grolifant.api.core.GradleSysEnvProvider
import org.ysb33r.grolifant.internal.core.Warnings
import org.ysb33r.grolifant.internal.v4.property.providers.GradleSysEnvProviderFactory
import org.ysb33r.grolifant.loadable.core.ProjectOperationsProxy

import java.util.concurrent.Callable
import java.util.function.Function

import static org.ysb33r.grolifant.api.core.LegacyLevel.PRE_5_0
import static org.ysb33r.grolifant.api.core.LegacyLevel.PRE_6_6

/**
 * An extension that can be added to a project by a plugin to aid in compatibility
 *
 * @since 1.0.0
 */
@CompileStatic
@Slf4j
class DefaultProjectOperations extends ProjectOperationsProxy {

    /**
     * Constructor that sets up a number of methods to be compatible across a wide range Gradle releases.
     *
     * @param project
     */
    @SuppressWarnings(['MethodSize', 'DuplicateStringLiteral'])
    DefaultProjectOperations(Project project) {
        super(project)

        if (!PRE_5_0) {
            Warnings.betterGrolifantAvailable('grolifant40')
        }
        if (!PRE_6_6) {
            Warnings.missingGrolifant60()
        }

        projectCacheDir = project.gradle.startParameter.projectCacheDir ?: project.file("${project.rootDir}/.gradle")
        this.providerFactory = project.providers
        this.fileOperations = ((ProjectInternal) project).services.get(FileOperations)
        this.propertyProvider = GradleSysEnvProviderFactory.loadInstance(project, this)
        archives = new org.ysb33r.grolifant.internal.v4.DefaultArchiveOperations(project)
        execs = new org.ysb33r.grolifant.internal.v4.DefaultExecOperations(project)
        fileTreeFactory = { Object base -> project.fileTree(base) }

        def projectLayout = project.layout
        this.buildDir = map(projectLayout.buildDirectory, { Directory it ->
            it.asFile
        } as Transformer<File, Directory>)
    }

    @Override
    CopySpec copySpec() {
        fileOperations.copySpec()
    }

    /**
     * Creates a new ConfigurableFileTree. The tree will have no base dir specified.
     *
     * @param base Base directory for file tree,
     * @return File tree.
     */
    @Override
    ConfigurableFileTree fileTree(Object base) {
        this.fileTreeFactory.apply(base)
    }

    /**
     * Build directory
     *
     * @return Provider to the build directory
     */
    @Override
    Provider<File> getBuildDir() {
        this.buildDir
    }

    /**
     * Safely resolve the stringy items as a path below build directory.
     *
     * @param stringy Any item that can be resolved to a string using
     * {@link org.ysb33r.grolifant.api.core.StringTools#stringize}
     * @return Provider to a file
     */
    @Override
    Provider<File> buildDirDescendant(Object stringy) {
        map(this.buildDir, { File it ->
            new File(it, stringTools.stringize(stringy))
        } as Transformer<File, File>)
    }

    /** Returns the project cache dir
     *
     * @return Location of cache directory
     */
    @Override
    File getProjectCacheDir() {
        this.projectCacheDir
    }

    /** Returns a provider.
     *
     * @param var1 Anything that adheres to a Callable including Groovy closures or Java lambdas.
     * @return Provider instance.
     */
    @Override
    public <T> Provider<T> provider(Callable<? extends T> var1) {
        providerFactory.provider(var1)
    }

    @Override
    Provider<String> environmentVariable(Object name, boolean configurationTimeSafety) {
        propertyProvider.environmentVariable(name, configurationTimeSafety)
    }

    @Override
    Provider<String> gradleProperty(Object name, boolean configurationTimeSafety) {
        propertyProvider.gradleProperty(name, configurationTimeSafety)
    }

    @Override
    Provider<String> systemProperty(Object name, boolean configurationTimeSafety) {
        propertyProvider.systemProperty(name, configurationTimeSafety)
    }

    @Override
    protected ArchiveOperationsProxy getArchiveOperations() {
        this.archives
    }

    @Override
    protected ExecOperationsProxy getExecOperations() {
        this.execs
    }

    @Override
    protected GradleSysEnvProvider getPropertyProvider() {
        this.propertyProvider
    }

    private final Function<Object, ConfigurableFileTree> fileTreeFactory
    private final File projectCacheDir
    private final ProviderFactory providerFactory
    private final Provider<File> buildDir
    private final ArchiveOperationsProxy archives
    private final ExecOperationsProxy execs
    private final FileOperations fileOperations
    private final GradleSysEnvProvider propertyProvider
}
