/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2022
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.internal.v4.jvm

import groovy.transform.CompileStatic
import org.gradle.api.Action
import org.gradle.api.tasks.Internal
import org.gradle.process.JavaExecSpec
import org.ysb33r.grolifant.api.core.CmdlineArgumentSpec
import org.ysb33r.grolifant.api.core.executable.ProcessExecutionSpec
import org.ysb33r.grolifant.api.core.ProjectOperations
import org.ysb33r.grolifant.api.core.jvm.JavaForkOptionsWithEnvProvider
import org.ysb33r.grolifant.api.core.jvm.JvmAppRunnerSpec
import org.ysb33r.grolifant.api.core.jvm.JvmEntryPoint
import org.ysb33r.grolifant.api.core.jvm.ModularitySpec
import org.ysb33r.grolifant.api.core.runnable.AbstractCmdlineArgumentSpec
import org.ysb33r.grolifant.internal.core.runnable.EnvironmentVariableProviders
import org.ysb33r.grolifant.internal.core.runnable.ProcessExecutionSpecProxy
import org.ysb33r.grolifant.internal.core.Transform

import java.util.function.BiFunction
import java.util.function.Function

/**
 * Internal class for common functionality related to {@code JavaExecSpec} configuration.
 *
 * Provider proxy functionality.
 *
 * @author Schalk W. Cronjé
 *
 * @since 2.0
 */
@CompileStatic
abstract class InternalAbstractJvmAppRunnerSpec implements JvmAppRunnerSpec {

    /**
     * Creates the JvmExecSpec on Gradle <6.4.
     *
     * @param po {@link ProjectOperations} instance
     * @param jfoProxyFactory Create a proxy instance for updating {@link org.gradle.process.JavaForkOptions}.
     * @param modularitySpecFactory Create a proxy instance for updating modularity specifications.
     */
    protected InternalAbstractJvmAppRunnerSpec(
        ProjectOperations po,
        BiFunction<JavaExecSpec, EnvironmentVariableProviders, JavaForkOptionsWithEnvProvider> jfoProxyFactory,
        Function<JavaExecSpec, ModularitySpec> modularitySpecFactory
    ) {
        this.javaExecSpec = po.jvmTools.javaExecSpec()
        this.arguments = new Arguments(po)
        this.projectOperations = po
        this.envProviders = new EnvironmentVariableProviders()
        this.jfoProxy = jfoProxyFactory.apply(this.javaExecSpec, envProviders)
        this.psProxy = new ProcessExecutionSpecProxy(this.javaExecSpec)
        this.jepProxy = new JvmEntryPointProxy(
            this.javaExecSpec,
            modularitySpecFactory.apply(this.javaExecSpec),
            this.projectOperations
        )
    }

    @Override
    JavaExecSpec copyTo(JavaExecSpec options) {
        copyToJavaExecSpec(options)
        options
    }

    @Override
    void configureForkOptions(Action<JavaForkOptionsWithEnvProvider> configurator) {
        configurator.execute(this.jfoProxy)
    }

    @Override
    void configureCmdline(Action<CmdlineArgumentSpec> configurator) {
        configurator.execute(this.arguments)
    }

    @Override
    void configureEntrypoint(Action<JvmEntryPoint> configurator) {
        configurator.execute(this.jepProxy)
    }

    @Override
    void configureProcess(Action<ProcessExecutionSpec> configurator) {
        configurator.execute(this.psProxy)
    }

    /**
     * Holds a copy of the internal {@link JavaExecSpec}
     */
    @Internal
    protected final JavaExecSpec javaExecSpec

    @Internal
    protected final Arguments arguments

    @Internal
    protected final ProjectOperations projectOperations

    @Internal
    protected final EnvironmentVariableProviders envProviders

    @Internal
    protected final JavaForkOptionsWithEnvProvider jfoProxy

    @Internal
    protected final JvmEntryPointProxy jepProxy

    @Internal
    protected final ProcessExecutionSpecProxy psProxy

    /**
     * Copies command arguments (non-JVM) target {@link JavaExecSpec}.
     *
     * @param target Target {@link JavaExecSpec}.
     */
    abstract void copyArguments(JavaExecSpec target)

    /**
     * Copgies debug options to target {@link JavaExecSpec}.
     * @param targetTarget {@link JavaExecSpec}.
     */
    abstract void copyDebugOptions(JavaExecSpec target)

    /**
     * Copies all settings to a target {@link JavaExecSpec}.
     *
     * @param target Target {@link JavaExecSpec}
     */
    protected void copyToJavaExecSpec(JavaExecSpec options) {
        this.javaExecSpec.copyTo(options)
        Transform.toList(envProviders.environmentProviders) {
            it.get()
        }.each {
            options.environment(it)
        }
        copyArguments(options)
        copyDebugOptions(options)
        jepProxy.copyTo(options)
        options.ignoreExitValue = javaExecSpec.ignoreExitValue

        if (javaExecSpec.standardOutput != null) {
            options.standardOutput = javaExecSpec.standardOutput
        }

        if (javaExecSpec.standardInput != null) {
            options.standardInput = javaExecSpec.standardInput
        }

        if (javaExecSpec.errorOutput != null) {
            options.errorOutput = javaExecSpec.errorOutput
        }
    }

    static protected class Arguments extends AbstractCmdlineArgumentSpec {
        Arguments(ProjectOperations po) {
            super(po.stringTools, po.providers)
        }
    }
}
