/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2023
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.internal.v4.jvm

import groovy.transform.CompileDynamic
import groovy.transform.CompileStatic
import groovy.transform.TypeChecked
import org.gradle.process.CommandLineArgumentProvider
import org.gradle.process.JavaExecSpec
import org.gradle.process.JavaForkOptions
import org.ysb33r.grolifant.api.core.LegacyLevel
import org.ysb33r.grolifant.api.core.ProjectOperations
import org.ysb33r.grolifant.api.core.jvm.JavaForkOptionsWithEnvProvider
import org.ysb33r.grolifant.internal.core.Transform
import org.ysb33r.grolifant.internal.core.runnable.EnvironmentVariableProviders

/**
 * Provides a class that can be populated with various fork options for Java
 * and which can then be used to copy to other methods in the Gradle API that provides a
 * {@link org.gradle.process.JavaForkOptions} in the parameters.
 *
 * Intended for Gradle 4.x
 *
 * @author Schalk W. Cronjé
 *
 * @since 2.0
 */
@CompileStatic
class InternalJvmAppRunnerSpec extends InternalAbstractJvmAppRunnerSpec {
    InternalJvmAppRunnerSpec(ProjectOperations po) {
        super(
            po,
            { JavaExecSpec jes, EnvironmentVariableProviders evp -> createJfoProxy(jes, evp) },
            { JavaExecSpec jes -> new FakeModularitySpec(po) }
        )
    }

    @Override
    void copyArguments(JavaExecSpec target) {
        if (LegacyLevel.PRE_4_6) {
            copyArgumentsInternalPre46(target)
        } else {
            copyArgumentsInternal(target)
        }
    }

    @Override
    void copyDebugOptions(JavaExecSpec target) {
    }

    /**
     * Builds a list of arguments by taking all the set arguments as well as the argument providers.
     *
     * THe main purpose of this method is to provide a list of arguments to be passed to a Worker instance.
     *
     * @return All application arguments
     */
    @Override
    protected List<String> buildArguments() {
        if (LegacyLevel.PRE_4_6) {
            buildArgumentsInternalPre46()
        } else {
            buildArgumentsInternal()
        }
    }

    private void copyArgumentsInternalPre46(JavaExecSpec options) {
        options.args(arguments.allArgs.get())
        commandLineProcessors.sort { lhs, rhs ->
            if (lhs.order == null && rhs.order == null) {
                0
            } else if (lhs.order == null) {
                1
            } else if (rhs.order == null) {
                -1
            } else {
                lhs.order <=> rhs.order
            }
        }.each {
            options.args(it.argumentSpec.allArgs.get())
        }
    }

    @CompileDynamic
    @TypeChecked
    private void copyArgumentsInternal(JavaExecSpec options) {
        options.args = arguments.args
        def cmdlineProviders = Transform.toList(arguments.commandLineArgumentProviders) { p ->
            new CommandLineArgumentProvider() {
                @Override
                Iterable<String> asArguments() {
                    p.get()
                }
            }
        }
        options.argumentProviders.addAll(cmdlineProviders)
        commandLineProcessors.each {
            final providedArgs = it.argumentSpec.commandLineArgumentProviders

            if (it.order) {
                final allArgs = it.argumentSpec.allArgs
                options.argumentProviders.add({ -> allArgs.get() } as CommandLineArgumentProvider)
            } else {
                options.args(it.argumentSpec.args)

                providedArgs.each { p ->
                    options.argumentProviders.add({ -> p.get() } as CommandLineArgumentProvider)
                }
            }
        }
    }

    private List<String> buildArgumentsInternalPre46() {
        final List<String> result = []
        final tmpSpec = projectOperations.jvmTools.javaExecSpec()
        copyArguments(tmpSpec)
        result.addAll(tmpSpec.args)
        result
    }

    @CompileDynamic
    @TypeChecked
    private List<String> buildArgumentsInternal() {
        final List<String> result = []
        final tmpSpec = projectOperations.jvmTools.javaExecSpec()
        copyArguments(tmpSpec)
        result.addAll(tmpSpec.args)
        result.addAll(tmpSpec.argumentProviders*.asArguments().flatten().toList() as List<String>)
        result
    }

    private static JavaForkOptionsWithEnvProvider createJfoProxy(JavaExecSpec spec, EnvironmentVariableProviders evp) {
        if (LegacyLevel.PRE_4_6) {
            createProxyPre46(spec, evp)
        } else if (LegacyLevel.PRE_5_6) {
            createProxyPre56(spec, evp)
        } else {
            createProxy(spec, evp)
        }
    }

    @CompileDynamic
    private static JavaForkOptionsWithEnvProvider createProxy(
        JavaExecSpec es,
        EnvironmentVariableProviders evp
    ) {
        new JavaForkOptionsWithEnvProviderProxy(es, evp)
    }

    @CompileDynamic
    private static JavaForkOptionsWithEnvProvider createProxyPre56(
        JavaExecSpec es,
        EnvironmentVariableProviders evp
    ) {
        new Pre56JavaForkOptionsWithEnvProviderProxy(es as JavaForkOptions, evp) as JavaForkOptionsWithEnvProvider
    }

    @CompileDynamic
    private static JavaForkOptionsWithEnvProvider createProxyPre46(
        JavaExecSpec es,
        EnvironmentVariableProviders evp
    ) {
        new Pre46JavaForkOptionsWithEnvProviderProxy(es, evp) as JavaForkOptionsWithEnvProvider
    }
}
