/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2023
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.internal.v4.jvm.worker;

import org.gradle.process.JavaExecSpec;
import org.gradle.workers.IsolationMode;
import org.gradle.workers.WorkerExecutor;
import org.ysb33r.grolifant.api.core.jvm.JvmAppRunnerSpec;
import org.ysb33r.grolifant.api.core.jvm.worker.WorkerAppParameterFactory;
import org.ysb33r.grolifant.api.core.jvm.worker.WorkerExecSpec;
import org.ysb33r.grolifant.api.core.jvm.worker.WorkerIsolation;
import org.ysb33r.grolifant.api.core.jvm.worker.WorkerPromise;
import org.ysb33r.grolifant.api.remote.worker.SerializableWorkerAppParameters;
import org.ysb33r.grolifant.api.remote.worker.WorkerAppExecutorFactory;

import java.io.File;
import java.util.Set;

import static org.ysb33r.grolifant.api.core.jvm.worker.WorkerIsolation.CLASSPATH;

/**
 *
 * Abstract base class for dealing with worker queue submissions on Gradle < 5.6.
 *
 * @author Schalk W. Cronjé
 *
 * @since 2.0
 */
abstract public class AbstractWorkerSubmitterJvmAppRunnerSpec implements JvmAppRunnerSpec {
    /**
     * Submits this to a worker queue using an appropriate isolation mode.
     *
     * @param isolationMode            Isolation mode which is either classpath isolated or out of process.
     * @param worker                   A worker execution instance.
     * @param workerAppExecutorFactory A factory instances that can create executor logic.
     * @param parameterFactory         A factory which can create parameters and populate them from a
     *                                 {@link JavaExecSpec}.
     */
    @Override
    public <P extends SerializableWorkerAppParameters> WorkerPromise submitToWorkQueue(
            WorkerIsolation isolationMode,
            WorkerExecutor worker,
            WorkerAppExecutorFactory<P> workerAppExecutorFactory,
            WorkerAppParameterFactory<P> parameterFactory
    ) {
        final Set<File> classpathFiles = getInternalJavaSpec().getClasspath().getFiles();
        final WorkerExecSpec workExecSpec = createInternalWorkerExecSpec();
        worker.submit(WorkerAppExecutorPre56.class, config -> {
                    if (isolationMode.equals(CLASSPATH)) {
                        config.setIsolationMode(IsolationMode.CLASSLOADER);
                        config.classpath(classpathFiles);
                    } else {
                        config.setIsolationMode(IsolationMode.PROCESS);
                        config.classpath(classpathFiles);
                        getInternalJavaSpec().copyTo(config.getForkOptions());
                    }

                    config.params(
                        parameterFactory.createAndConfigure(workExecSpec),
                        workerAppExecutorFactory.createExecutor()
                    );
                }
        );
        return () -> worker.await();
    }

    protected abstract JavaExecSpec getInternalJavaSpec();

    protected abstract WorkerExecSpec createInternalWorkerExecSpec();
}
