/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2024
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.loadable.v4

import groovy.transform.CompileStatic
import org.gradle.api.Transformer
import org.gradle.api.model.ObjectFactory
import org.gradle.api.provider.Provider
import org.gradle.api.provider.ProviderFactory
import org.ysb33r.grolifant.loadable.core.ProviderToolsProxy

/**
 * Safely deal with Providers down to Gradle 4.0.
 *
 * @author Schalk W. Cronjé
 *
 * @since 1.1
 */
@CompileStatic
class DefaultProviderTools extends ProviderToolsProxy {

    DefaultProviderTools(ProviderFactory providerFactory, ObjectFactory objectFactory) {
        super(providerFactory, objectFactory)
    }

    /**
     * Allow flatMap functionality for providers even before Gradle 5.0.
     *
     * @param provider Existing provider.
     * @param transformer Transform one provider to another.
     * @param <S >     Return type of new provider.
     * @param <T >     Return type of existing provider.
     * @return New provider.
     *
     * @since 1.2
     */
    @Override
    public <S, T> Provider<S> flatMap(
        Provider<T> provider,
        Transformer<? extends Provider<? extends S>, ? super T> transformer
    ) {
        providerFactory.provider { ->
            if (provider.present) {
                def providerValue = provider.get()
                def transformedValue = transformer.transform(providerValue)
                transformedValue.getOrNull()
            } else {
                null
            }
        }
    }

    /**
     * Allow orElse functionality prior to Gradle 5.6.
     *
     * Returns a Provider whose value is the value of this provider, if present, otherwise the given default value.
     *
     * @param provider Original provider.
     * @param value The default value to use when this provider has no value.
     * @param <T >   Provider type.
     * @return Provider value or default value.
     *
     * @since 1.2
     */
    @Override
    public <T> Provider<T> orElse(Provider<T> provider, T value) {
        providerFactory.provider { ->
            provider.getOrElse(value)
        }
    }

    /**
     * Allow orElse functionality prior to Gradle 5.6.
     * Returns a Provider whose value is the value of this provider, if present, otherwise uses the value from
     * the given provider, if present.
     *
     * @param provider Original provider
     * @param elseProvider The provider whose value should be used when this provider has no value.
     * @param <T >   Provider type
     * @return Provider chain.
     *
     * @since 1.2
     */
    @Override
    public <T> Provider<T> orElse(Provider<T> provider, Provider<? extends T> elseProvider) {
        providerFactory.provider { ->
            provider.present ? provider.get() : elseProvider.get()
        }
    }

    /**
     * Creates a provider that can resolve the three providers in order.
     * If the first is not present, it will attempt to resolve the second and then the third.
     *
     * @param one First provider.
     * @param two Second provider.
     * @param three Third provider.
     * @return Combined resolver.
     *
     * @since 2.0
     */
    @Override
    Provider<String> resolveOrderly(Provider<String> one, Provider<String> two, Provider<String> three) {
        providerFactory.provider { ->
            one.present ? one.get() : (two.present ? two.get() : three.getOrNull())
        }
    }
}
