/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2024
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.loadable.v4

import groovy.transform.CompileDynamic
import groovy.transform.CompileStatic
import org.gradle.api.Action
import org.gradle.api.DefaultTask
import org.gradle.api.Project
import org.gradle.api.Task
import org.gradle.api.tasks.TaskInputFilePropertyBuilder
import org.ysb33r.grolifant.api.core.LegacyLevel
import org.ysb33r.grolifant.api.core.ProjectOperations
import org.ysb33r.grolifant.api.core.TaskInputFileOptions
import org.ysb33r.grolifant.api.errors.NotSupportedException
import org.ysb33r.grolifant.loadable.core.TaskToolsProxy

import static org.ysb33r.grolifant.api.core.LegacyLevel.PRE_4_7
import static org.ysb33r.grolifant.api.core.LegacyLevel.PRE_4_8
import static org.ysb33r.grolifant.api.core.LegacyLevel.PRE_4_9
import static org.ysb33r.grolifant.api.core.LegacyLevel.PRE_5_0
import static org.ysb33r.grolifant.api.core.TaskInputFileOptions.OPTIONAL
import static org.ysb33r.grolifant.api.core.TaskInputFileOptions.SKIP_WHEN_EMPTY

@CompileStatic
class DefaultTaskTools extends TaskToolsProxy {

    DefaultTaskTools(ProjectOperations incompleteReference, Project project) {
        super(incompleteReference, project)
    }

    /**
     * Registers a task in a lazy-manner.
     * <p>
     * On Gradle 4.9 or earlier it create the task.
     *
     * @param taskName Name of task to register. Task must have been registered or configured previously.
     * @param taskType Type of task.
     * @param configurator Configurating action.
     * @param <T >     Type of task.
     */
    @Override
    @CompileDynamic
    public <T extends DefaultTask> void register(String taskName, Class<T> taskType, Action<T> configurator) {
        if (PRE_4_9) {
            configurator.execute(tasks.create(taskName, taskType))
        } else {
            tasks.register(taskName, taskType).configure(configurator)
        }
    }

    /**
     * Registers a task in a lazy-manner.
     *
     * @param taskName Name of task to register. Task must have been registered or configured previously.
     * @param taskType Type of task.
     * @param args C-tor arguments.
     * @param configurator Configurating action.
     * @param <T>                   Type of task.
     */
    @Override
    public <T extends DefaultTask> void register(
        String taskName,
        Class<T> taskType,
        Iterable<Object> args,
        Action<T> configurator
    ) {
        final argsList = args.toList()
        if (PRE_4_7 && !argsList.empty) {
            throw new NotSupportedException('Constructor arguments cannot be provided on Gradle < 4.7')
        } else if (PRE_4_9) {
            configurator.execute(tasks.create(taskName, taskType, argsList as Object[]))
        } else {
            final task = tasks.register(taskName, taskType, argsList as Object[])
            task.configure(configurator)
        }
    }

    /**
     * Configures a task, preferably in a lazy-manner.
     * <p>
     * On Gradle 4.9 or earlier it will use {@code getByName} internally.
     *
     * @param taskName Name of task to configure.  Task must have been registered or configured previously.
     * @param configurator Configurating action.
     */
    @Override
    @CompileDynamic
    void named(String taskName, Action<Task> configurator) {
        if (PRE_4_9) {
            configurator.execute(tasks.getByName(taskName))
        } else if (PRE_5_0) {
            tasks.named(taskName).configure(configurator)
        } else {
            tasks.named(taskName, configurator)
        }
    }

    /**
     * Adds a configuration for a task, for when a task is created.
     *
     * @param taskName Name of task to configure.
     * @param configurator Configurating action.
     */
    @Override
    void whenNamed(String taskName, Action<Task> configurator) {
        if (PRE_4_9) {
            tasks.whenTaskAdded { Task it ->
                if (it.name == taskName) {
                    configurator.execute(it)
                }
            }
        } else {
            tasks.configureEach { Task task ->
                if (task.name == taskName) {
                    configurator.execute(task)
                }
            }
        }
    }

    /**
     * Configures a task, preferably in a lazy-manner.
     * <p>
     * On Gradle 4.9 or earlier it will use {@code getByName} internally.
     *
     * @param taskName Name of task to configure. Task must have been registered or configured previously.
     * @param taskType Type of task.
     * @param configurator Configurating action.
     * @param <T >               Type of task.
     */
    @Override
    public <T extends DefaultTask> void named(String taskName, Class<T> taskType, Action<T> configurator) {
        if (PRE_4_9) {
            configurator.execute(taskType.cast(tasks.getByName(taskName)))
        } else {
            tasks.named(taskName).configure {
                configurator.execute(taskType.cast(it))
            }
        }
    }

    /**
     * Adds a configuration for a task, for when a task is created.
     *
     * @param taskName Name of task to configure.
     * @param taskType Type of task.
     * @param configurator Configurating action.
     * @param <T >                  Type of task.
     */
    @Override
    @CompileDynamic
    public <T extends DefaultTask> void whenNamed(String taskName, Class<T> taskType, Action<T> configurator) {
        if (LegacyLevel.PRE_4_9) {
            tasks.whenTaskAdded { Task it ->
                if (it.name == taskName) {
                    configurator.execute(it)
                }
            }
        } else {
            tasks.withType(taskType) { T task ->
                if (task.name == taskName) {
                    configurator.execute(task)
                }
            }
        }
    }

    /**
     * Creates an input to a task that is based upon a cooleciton of files.
     *
     * @param inputsBuilder The property builder
     * @param options Additional options to assign to the task.
     * Some options might be ignored, depending on the version of Gradle.
     */
    @Override
    protected void createTaskInputsFileEntry(
        TaskInputFilePropertyBuilder inputsBuilder,
        List<TaskInputFileOptions> options
    ) {
        options.forEach {
            switch (it) {
                case SKIP_WHEN_EMPTY:
                    inputsBuilder.skipWhenEmpty()
                    break
                case OPTIONAL:
                    inputsBuilder.optional()
                    break
            }
        }
    }

    /**
     * Whether this is a {@code TaskProvider<?>}.
     *
     * For Gradle < 4.8 it will always return {@code false}.
     *
     * @param o Object to evaluate
     * @return {@code true} is an instance of {@link org.gradle.api.tasks.TaskProvider}.
     */
    @Override
    protected boolean isTaskProvider(Object o) {
        isGradleTaskProvider(o)
    }

    @CompileDynamic
    private boolean isGradleTaskProvider(Object o) {
        if (PRE_4_8) {
            false
        } else {
            o instanceof org.gradle.api.tasks.TaskProvider
        }
    }
}
