/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2024
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.internal.v4.executable

import groovy.transform.CompileStatic
import org.gradle.api.tasks.Internal
import org.gradle.process.BaseExecSpec
import org.gradle.process.ExecSpec
import org.gradle.process.ProcessForkOptions
import org.ysb33r.grolifant.api.core.ProjectOperations
import org.ysb33r.grolifant.api.core.runnable.AbstractCmdlineArgumentSpec

/**
 * Fake execution specification for Gradle 4.0 - 4.5.1
 *
 * @author Schalk W. Cronjé
 *
 * @since 2.0
 */
@CompileStatic
@SuppressWarnings('MethodCount')
class Pre46FakeExecSpec {

    Pre46FakeExecSpec(ProjectOperations projectOperations) {
        this.projectOperations = projectOperations
        this.arguments = new Arguments(projectOperations)
        this.stdin = System.in
        this.stdout = System.out
        this.stderr = System.err
        this.wd = projectOperations.projectDir
    }

    /**
     * Adds arguments for the command to be executed.
     *
     * @param args args for the command
     * @return this
     */

    ExecSpec args(Object... args) {
        arguments.args(args)
        me()
    }

    /**
     * Adds arguments for the command to be executed.
     *
     * @param args args for the command
     * @return this
     */

    ExecSpec args(Iterable<?> args) {
        arguments.args(args)
        me()
    }

    /**
     * Sets the full command line, including the executable to be executed plus its arguments.
     *
     * @param args the command plus the args to be executed
     * @return this
     */

    ExecSpec commandLine(Object... args) {
        commandLine(args as List)
    }

    /**
     * Sets the full command line, including the executable to be executed plus its arguments.
     *
     * @param args the command plus the args to be executed
     * @return this
     */

    ExecSpec commandLine(Iterable<?> args) {
        switch (args.size()) {
            case 0:
                break
            case 1:
                executable(args[0])
                arguments.args = []
                break
            default:
                executable(args[0])
                arguments.args = (args.drop(1))
        }
        me()
    }

    /**
     * Returns the arguments for the command to be executed. Defaults to an empty list.
     */

    List<String> getArgs() {
        arguments.args
    }

    /**
     * Returns the full command line, including the executable plus its arguments.
     *
     * @return The full command line, including the executable plus its arguments
     */

    List<String> getCommandLine() {
        projectOperations.stringTools.stringizeDropNull([executable] + arguments.args)
    }

    /**
     * Returns the name of the executable to use.
     *
     * @return The executable.
     */

    String getExecutable() {
        projectOperations.stringTools.stringize(exe)
    }

    /**
     * Returns the working directory for the process. Defaults to the project directory.
     *
     * @return The working directory. Never returns null.
     */

    File getWorkingDir() {
        projectOperations.fsOperations.file(wd)
    }

    /**
     * Copies these options to the given target options.
     *
     * @param target Another implementation {@link org.gradle.process.ProcessForkOptions}.
     * @return {@code this}
     */

    ProcessForkOptions copyTo(ProcessForkOptions target) {
        target.environment = environment
        target.executable = executable
        target.workingDir = workingDir
        (ProcessForkOptions) me()
    }

    /**
     * Adds an environment variable to the environment for this process.
     *
     * @param name Environment variable name
     * @param value Environment variable value
     * @return {@code this}
     */

    ExecSpec environment(String name, Object value) {
        this.env.put(name, value)
        this as ExecSpec
    }

    /**
     * Adds some environment variables to the environment for this process.
     *
     * @param environmentVariables Map of environment variables
     * @return {@code this}
     */

    ExecSpec environment(Map<String, ?> environmentVariables) {
        this.env.putAll(environmentVariables)
        this as ExecSpec
    }

    /**
     * Sets the name of the executable to use.
     *
     * @param executable Name of executable. This could be resolved to a {@link String}, {@link File}
     *                   or {@link java.nio.file.Path} before execution
     * @return {@code this}
     */

    ExecSpec executable(Object executable) {
        this.exe = executable
        me()
    }

    /**
     * The environment variables to use for the process.
     *
     * @return Environment, but not yet resolved from original values.
     */

    Map<String, Object> getEnvironment() {
        this.env
    }

    /**
     * Returns the output stream to consume standard error from the process executing the command.
     *
     * @return Error stream.
     */

    @Internal
    OutputStream getErrorOutput() {
        this.stderr
    }

    /**
     * Returns the standard input stream for the process executing the command.
     *
     * @return Standard input stream.
     */

    @Internal
    InputStream getStandardInput() {
        this.stdin
    }

    /**
     * Returns the output stream to consume standard output from the process executing the command.
     *
     * @return Standard output stream.
     */

    @Internal
    OutputStream getStandardOutput() {
        this.stdout
    }

    /**
     * Tells whether a non-zero exit value is ignored, or an exception thrown.
     *
     * @return {@code true} if return code is ignored.
     */

    @Internal
    boolean isIgnoreExitValue() {
        this.ignoreExit
    }

    /**
     * Sets the arguments for the command to be executed.
     *
     * @param args args for the command
     * @return this
     * @since 4.0
     */

    ExecSpec setArgs(List<String> args) {
        arguments.args = args
        me()
    }

    /**
     * Sets the arguments for the command to be executed.
     *
     * @param args args for the command
     * @return this
     */

    ExecSpec setArgs(Iterable<?> args) {
        arguments.args = args
        me()
    }

    /**
     * Sets the full command line, including the executable to be executed plus its arguments.
     *
     * @param args the command plus the args to be executed
     * @since 4.0
     */

    void setCommandLine(List<String> args) {
        commandLine(args)
    }

    /**
     * Sets the full command line, including the executable to be executed plus its arguments.
     *
     * @param args the command plus the args to be executed
     */

    void setCommandLine(Object... args) {
        commandLine(args)
    }

    /**
     * Sets the full command line, including the executable to be executed plus its arguments.
     *
     * @param args the command plus the args to be executed
     */

    void setCommandLine(Iterable<?> args) {
        commandLine(args)
    }

    /**
     * Replaces the existing environment variables to use for the process with a new collection.
     *
     * @param environmentVariables Environment variables
     */

    void setEnvironment(Map<String, ?> environmentVariables) {
        this.env.clear()
        this.env.putAll(environmentVariables)
    }

    /**
     * Sets the output stream to consume standard error from the process executing the command.
     *
     * @param outputStream Redirect error output to this stream.
     * @return {@code this}.
     */

    BaseExecSpec setErrorOutput(OutputStream outputStream) {
        this.stderr = outputStream
        (BaseExecSpec) me()
    }

    /**
     * Sets the name of the executable to use.
     * <p>
     * This will be resolved to a {@link File}, {@link String} or {@link java.nio.file.Path} before execution.
     * </p>
     *
     * @param executable Executable, optionally including a path.
     */

    void setExecutable(Object executable) {
        this.exe = executable
    }

    /**
     * Sets the name of the executable to use.
     *
     * @param executable The executable. Must not be null.
     * @since 4.0
     */

    void setExecutable(String executable) {
        this.exe = executable
    }

    /**
     * Sets whether a non-zero exit value is ignored, or an exception thrown.
     *
     * @param ignoreExitValue {@code true} to prevent exception from being thrown on error.
     * @return {@code this}.
     */

    BaseExecSpec setIgnoreExitValue(boolean ignoreExitValue) {
        this.ignoreExit = ignoreExitValue
        (BaseExecSpec) me()
    }

    /**
     * Sets the standard input stream for the process executing the command.
     *
     * @param inputStream Redirect input to this stream.
     * @return {@code this}.
     */

    BaseExecSpec setStandardInput(InputStream inputStream) {
        this.stdin = inputStream
        (BaseExecSpec) me()
    }

    /**
     * Sets the output stream to consume standard output from the process executing the command.
     *
     * @param outputStream Redirect standard output to this stream.
     * @return {@code this}.
     */

    BaseExecSpec setStandardOutput(OutputStream outputStream) {
        this.stdout = outputStream
        (BaseExecSpec) me()
    }

    /**
     * Sets the working directory for the process.
     *
     * <p>Directory will be resolved to a {@link File} or {@link java.nio.file.Path} before execution.</p>
     *
     * @param dir Working directory
     */

    void setWorkingDir(Object dir) {
        this.wd = dir
    }

    void setWorkingDir(File file) {
        this.wd = file
    }

    ProcessForkOptions workingDir(Object o) {
        this.wd = o
        (ProcessForkOptions) me()
    }

    protected ProjectOperations getProjectOperations() {
        this.projectOperations
    }

    protected ExecSpec me() {
        this as ExecSpec
    }

    private final ProjectOperations projectOperations
    private final Arguments arguments
    private final Map<String, Object> env = new TreeMap<String, Object>()
    private InputStream stdin
    private OutputStream stdout
    private OutputStream stderr
    private Object exe
    private Object wd
    private boolean ignoreExit = false

    static private class Arguments extends AbstractCmdlineArgumentSpec {
        Arguments(ProjectOperations po) {
            super(po.stringTools, po.providers)
        }
    }
}
