/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2024
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.internal.v4.jvm

import groovy.transform.CompileStatic
import org.gradle.api.Action
import org.gradle.api.tasks.Internal
import org.gradle.process.JavaExecSpec
import org.ysb33r.grolifant.api.core.ClosureUtils
import org.ysb33r.grolifant.api.core.CmdlineArgumentSpec
import org.ysb33r.grolifant.api.core.ProjectOperations
import org.ysb33r.grolifant.api.core.executable.CmdLineArgumentSpecEntry
import org.ysb33r.grolifant.api.core.executable.ProcessExecutionSpec
import org.ysb33r.grolifant.api.core.jvm.JavaForkOptionsWithEnvProvider
import org.ysb33r.grolifant.api.core.jvm.JvmAppRunnerSpec
import org.ysb33r.grolifant.api.core.jvm.JvmEntryPoint
import org.ysb33r.grolifant.api.core.jvm.ModularitySpec
import org.ysb33r.grolifant.api.core.jvm.worker.WorkerExecSpec
import org.ysb33r.grolifant.api.core.runnable.AbstractCmdlineArgumentSpec
import org.ysb33r.grolifant.internal.core.Transform
import org.ysb33r.grolifant.internal.core.runnable.DefaultArguments
import org.ysb33r.grolifant.internal.core.runnable.EnvironmentVariableProviders
import org.ysb33r.grolifant.internal.core.runnable.ProcessExecutionSpecProxy
import org.ysb33r.grolifant.internal.v4.jvm.worker.AbstractWorkerSubmitterJvmAppRunnerSpec

import java.security.MessageDigest
import java.util.function.BiFunction
import java.util.function.Function

/**
 * Internal class for common functionality related to {@code JavaExecSpec} configuration.
 *
 * Provider proxy functionality.
 *
 * @author Schalk W. Cronjé
 *
 * @since 2.0
 */
@CompileStatic
abstract class InternalAbstractJvmAppRunnerSpec extends AbstractWorkerSubmitterJvmAppRunnerSpec
    implements JvmAppRunnerSpec {

    /**
     * Creates the JvmExecSpec on Gradle <6.4.
     *
     * @param po {@link ProjectOperations} instance
     * @param jfoProxyFactory Create a proxy instance for updating {@link org.gradle.process.JavaForkOptions}.
     * @param modularitySpecFactory Create a proxy instance for updating modularity specifications.
     */
    protected InternalAbstractJvmAppRunnerSpec(
        ProjectOperations po,
        BiFunction<JavaExecSpec, EnvironmentVariableProviders, JavaForkOptionsWithEnvProvider> jfoProxyFactory,
        Function<JavaExecSpec, ModularitySpec> modularitySpecFactory
    ) {
        this.cmdlineProcessors = [] as SortedSet<CmdLineArgumentSpecEntry>
        this.javaExecSpec = po.jvmTools.javaExecSpec()
        this.arguments = new Arguments(po)
        this.projectOperations = po
        this.envProviders = new EnvironmentVariableProviders()
        this.jfoProxy = jfoProxyFactory.apply(this.javaExecSpec, envProviders)
        this.psProxy = new ProcessExecutionSpecProxy(this.javaExecSpec)
        this.jepProxy = new JvmEntryPointProxy(
            this.javaExecSpec,
            modularitySpecFactory.apply(this.javaExecSpec),
            this.projectOperations
        )

        addCommandLineProcessor(DEFAULT_BLOCK, 0, new DefaultArguments(po))
    }

    @Override
    JavaExecSpec copyTo(JavaExecSpec options) {
        copyToJavaExecSpec(options)
        options
    }

    @Override
    void configureForkOptions(Action<JavaForkOptionsWithEnvProvider> configurator) {
        configurator.execute(this.jfoProxy)
    }

    @Override
    void configureForkOptions(@DelegatesTo(JavaForkOptionsWithEnvProvider) Closure<?> configurator) {
        ClosureUtils.configureItem(this.jfoProxy, configurator)
    }

    @Override
    void configureCmdline(Action<CmdlineArgumentSpec> configurator) {
        configurator.execute(this.arguments)
    }

    @Override
    void configureCmdline(@DelegatesTo(CmdlineArgumentSpec) Closure<?> configurator) {
        ClosureUtils.configureItem(this.arguments, configurator)
    }

    @Override
    void configureEntrypoint(Action<JvmEntryPoint> configurator) {
        configurator.execute(this.jepProxy)
    }

    @Override
    void configureEntrypoint(@DelegatesTo(JvmEntryPoint) Closure<?> configurator) {
        ClosureUtils.configureItem(this.jepProxy, configurator)
    }

    @Override
    void configureProcess(Action<ProcessExecutionSpec> configurator) {
        configurator.execute(this.psProxy)
    }

    @Override
    void configureProcess(@DelegatesTo(ProcessExecutionSpec) Closure<?> configurator) {
        ClosureUtils.configureItem(this.psProxy, configurator)
    }

    /**
     * Adds a command-line processor that will process command-line arguments in a specific order.
     *
     * For instance in a script, one want to proccess the exe args before the script args.
     *
     * In a system that has commands and subcommands, one wants to process this in the order of exe args, command args,
     * and then subcommand args.
     *
     * This method allows the implementor to control the order of processing  for all the groupings.
     *
     * @param name Name of command-line processor.
     * @param order Order in queue to process.
     * @param processor The specific grouping.
     */
    @Override
    void addCommandLineProcessor(String name, Integer order, CmdlineArgumentSpec processor) {
        this.cmdlineProcessors.add(new CmdLineArgumentSpecEntry(name, order, processor))
    }

    /**
     * Provides direct access to the list of command-line processors.
     *
     * In many cases there will only be one item in the list which is for providing arguments to the executable
     * itself. Some implementations will have more. Implementors can use this interface to manipulate order of
     * evaluation.
     *
     * @return Collection of command-line processors. Can be empty, but never {@code null}.
     */
    @Override
    Collection<CmdLineArgumentSpecEntry> getCommandLineProcessors() {
        this.cmdlineProcessors
    }

    /**
     * A unique string which determines whether there were any changes.
     *
     * @return Signature string
     */
    @Override
    String getExecutionSignature() {
        MessageDigest.getInstance('SHA-256').digest(executionParameters.toString().bytes)
            .encodeHex().toString()
    }

    /**
     * Holds a copy of the internal {@link JavaExecSpec}
     */
    @Internal
    protected final JavaExecSpec javaExecSpec

    @Internal
    protected final Arguments arguments

    @Internal
    protected final ProjectOperations projectOperations

    @Internal
    protected final EnvironmentVariableProviders envProviders

    @Internal
    protected final JavaForkOptionsWithEnvProvider jfoProxy

    @Internal
    protected final JvmEntryPointProxy jepProxy

    @Internal
    protected final ProcessExecutionSpecProxy psProxy

    /**
     * Copies command arguments (non-JVM) target {@link JavaExecSpec}.
     *
     * @param target Target {@link JavaExecSpec}.
     */
    abstract void copyArguments(JavaExecSpec target)

    /**
     * Copgies debug options to target {@link JavaExecSpec}.
     * @param targetTarget {@link JavaExecSpec}.
     */
    abstract void copyDebugOptions(JavaExecSpec target)

    /**
     * Builds a list of arguments by taking all the set arguments as well as the argument providers.
     *
     * THe main purpose of this method is to provide a list of arguments to be passed to a Worker instance.
     *
     * @return All application arguments
     */
    abstract protected List<String> buildArguments()

    /**
     * Copies all settings to a target {@link JavaExecSpec}.
     *
     * TResolved environment to strings before copying as Gradle's API does not deal with certain kinds of providers
     * or recursive resolving.
     *
     * @param target Target {@link JavaExecSpec}
     */
    protected void copyToJavaExecSpec(JavaExecSpec options) {
        this.javaExecSpec.copyTo(options)
        options.environment(projectOperations.stringTools.stringizeValues(this.javaExecSpec.environment))
        Transform.toList(envProviders.environmentProviders) {
            it.get()
        }.each {
            options.environment(it)
        }
        copyArguments(options)
        copyDebugOptions(options)
        jepProxy.copyTo(options)
        options.ignoreExitValue = javaExecSpec.ignoreExitValue

        if (javaExecSpec.standardOutput != null) {
            options.standardOutput = javaExecSpec.standardOutput
        }

        if (javaExecSpec.standardInput != null) {
            options.standardInput = javaExecSpec.standardInput
        }

        if (javaExecSpec.errorOutput != null) {
            options.errorOutput = javaExecSpec.errorOutput
        }
    }

    /**
     * Loads executions parameters from the current execution specification.
     * <p>
     *     The primary purpose of this method is to build a map for use by {@link #getExecutionSignature}.
     *     The default implementation will use the executable path and the arguments.
     *     </p>
     * @return Execution parameters.
     */
    @Internal
    protected Map<String, ?> getExecutionParameters() {
        final st = projectOperations.stringTools
        [
            exe             : st.stringizeOrNull(javaExecSpec.executable),
            args1           : st.stringizeDropNull(javaExecSpec.jvmArgs),
            args3           : st.stringizeDropNull(javaExecSpec.args),
            systemProperties: st.stringizeValues(javaExecSpec.systemProperties),
            classpath       : javaExecSpec.classpath.asPath,
            main            : st.stringizeOrNull(javaExecSpec.main)
        ]
    }

    @Override
    protected JavaExecSpec getInternalJavaSpec() {
        javaExecSpec
    }

    @Override
    protected WorkerExecSpec createInternalWorkerExecSpec() {
        new InternalWorkerExecSpec(this)
    }

    static protected class Arguments extends AbstractCmdlineArgumentSpec {
        Arguments(ProjectOperations po) {
            super(po.stringTools, po.providers)
        }
    }

    static private class InternalWorkerExecSpec implements WorkerExecSpec {
        InternalWorkerExecSpec(InternalAbstractJvmAppRunnerSpec thisRef) {
            this.ref = thisRef
        }

        @Override
        JavaExecSpec getJavaExecSpec() {
            ref.javaExecSpec
        }

        @Override
        JvmEntryPoint getJvmEntrypoint() {
            ref.jepProxy
        }

        @Override
        List<String> getApplicationArguments() {
            ref.buildArguments()
        }

        private final InternalAbstractJvmAppRunnerSpec ref
    }

    private final SortedSet<CmdLineArgumentSpecEntry> cmdlineProcessors
}
