/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2024
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant5.loadable.v7

import groovy.transform.CompileDynamic
import groovy.transform.CompileStatic
import org.gradle.api.Project
import org.gradle.api.provider.Provider
import org.ysb33r.grolifant5.loadable.core.LoadableVersion
import org.ysb33r.grolifant5.loadable.core.ProjectToolsProxy

import javax.inject.Inject

import static org.ysb33r.grolifant5.api.core.LegacyLevel.PRE_7_6

/**
 * @since 2.0
 */
@CompileStatic
class DefaultProjectTools extends ProjectToolsProxy {

    @Inject
    DefaultProjectTools(Project project) {
        super(
            project,
            PRE_7_6 ? defaultVersionProvider71to75(project) : defaultVersionProvider76(project),
            LoadableVersion.V7
        )

        this.projectGroup = propertyByName(project, 'group').orElse(UNSPECIFIED_PROJECT_GROUP)
    }

    /**
     * Lazy-evaluated project group.
     *
     * @return provider to project group
     */
    @Override
    Provider<String> getGroupProvider() {
        this.projectGroup
    }

    static private Provider<String> propertyByName(Project project, String name) {
        project.providers.gradleProperty(name).forUseAtConfigurationTime()
    }

    static private Provider<String> defaultVersionProvider71to75(Project project) {
        propertyByName(project, VERSION_PROPERTY)
            .orElse(project.provider { -> project.version?.toString() })
            .orElse(UNSPECIFIED_PROJECT_VERSION)
        project.provider { -> project.version?.toString() }
    }

    @CompileDynamic
    static private Provider<String> defaultVersionProvider76(Project project) {
        if (project.gradle.startParameter.configurationCacheRequested) {
            Object cachedProjectVersion = project.version
            propertyByName(project, VERSION_PROPERTY)
                .orElse(project.provider { -> cachedProjectVersion?.toString() })
                .orElse(UNSPECIFIED_PROJECT_VERSION)
        } else {
            project.provider { -> project.version?.toString() }
                .orElse(UNSPECIFIED_PROJECT_VERSION)
        }
    }

    private final Provider<String> projectGroup

    private static final String VERSION_PROPERTY = 'version'
}
