/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2024
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant5.internal.v7.runnable

import groovy.transform.CompileDynamic
import groovy.transform.CompileStatic
import org.gradle.api.provider.Provider
import org.gradle.api.provider.ProviderFactory
import org.gradle.process.ExecResult
import org.ysb33r.grolifant5.api.core.LegacyLevel
import org.ysb33r.grolifant5.api.core.runnable.ExecOutput
import org.ysb33r.grolifant5.api.core.runnable.ExecStreamContent
import org.ysb33r.grolifant5.api.errors.NotSupportedException

/**
 * Wraps the output from an execution on Gradle 7.x.
 *
 * @author Schalk W. Cronjé
 *
 * @since 5.0
 */
@CompileStatic
class DefaultExecOutput implements ExecOutput {

    static ExecOutput fromResult(ProviderFactory providers, Object execResult) {
        if (LegacyLevel.PRE_7_5) {
            if (execResult instanceof ExecResult) {
                final Provider<String> mockContent = providers.provider { -> '' }
                final sc = new StreamContent(mockContent, mockContent.map { it.bytes })
                new DefaultExecOutput(
                    providers.provider { -> (ExecResult) execResult },
                    sc,
                    sc
                )
            } else {
                throw new NotSupportedException('On Gradle < 7.5 an ExecResult is expected')
            }
        } else {
            processResult(execResult)
        }
    }

    static ExecOutput fromResult(
        Provider<ExecResult> result,
        Provider<byte[]> outBytes,
        Provider<String> outText,
        Provider<byte[]> errBytes,
        Provider<String> errText
    ) {
        new DefaultExecOutput(
            result,
            new StreamContent(outText, outBytes),
            new StreamContent(errText, errBytes)
        )
    }
    final Provider<ExecResult> result
    final ExecStreamContent standardOutput
    final ExecStreamContent standardError

    private DefaultExecOutput(
        Provider<ExecResult> result,
        ExecStreamContent out,
        ExecStreamContent err
    ) {
        this.result = result
        this.standardOutput = out
        this.standardError = err
    }

    @CompileDynamic
    private static DefaultExecOutput processResult(Object execResult) {
        if (execResult.class.canonicalName == 'org.gradle.process.ExecOutput') {
            new DefaultExecOutput(
                execResult.result,
                new StreamContent(execResult.standardOutput.asText, execResult.standardOutput.asBytes),
                new StreamContent(execResult.standardError.asText, execResult.standardError.asBytes)
            )
        } else {
            throw new NotSupportedException('Requires org.gradle.process.ExecOutput')
        }
    }

    private static class StreamContent implements ExecStreamContent {
        final Provider<String> asText
        final Provider<byte[]> asBytes

        StreamContent(
            final Provider<String> asText,
            final Provider<byte[]> asBytes
        ) {
            this.asText = asText
            this.asBytes = asBytes
        }
    }
}
