/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2025
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant5.loadable.v7

import groovy.transform.CompileStatic
import org.gradle.api.Action
import org.gradle.api.Project
import org.gradle.api.provider.Provider
import org.ysb33r.grolifant5.api.core.ClosureUtils
import org.ysb33r.grolifant5.api.core.GrolifantProblemReporter
import org.ysb33r.grolifant5.internal.core.services.LegacyProblemDetails
import org.ysb33r.grolifant5.internal.core.services.LegacyWarningReporterService

import javax.inject.Inject

/**
 * Implementation for the problem reporter for Gradle 7.x
 *
 * @author Schalk W. Cronjé
 *
 * @since 5.4
 */
@CompileStatic
class DefaultGrolifantProblemReporter implements GrolifantProblemReporter {

    @Inject
    DefaultGrolifantProblemReporter(Project project) {
        this.problemReporter = LegacyWarningReporterService.registerIfAbsent(project)
    }

    @Override
    void report(ProblemId problemId, Action<ProblemSpec> config) {
        final spec = new LegacyProblemDetails()
        config.execute(spec)
        problemReporter.get().report(problemId, spec)
    }

    @Override
    void report(ProblemId problemId, @DelegatesTo(ProblemSpec) Closure<?> config) {
        final spec = new LegacyProblemDetails()
        ClosureUtils.configureItem(spec, config)
        problemReporter.get().report(problemId, spec)
    }

    @Override
    void report(String problemName, String displayName, ProblemGroup problemGroup, Action<ProblemSpec> config) {
        report(ProblemId.of(problemName, displayName, problemGroup), config)
    }

    @Override
    void report(
        String problemName,
        String displayName,
        ProblemGroup problemGroup,
        @DelegatesTo(ProblemSpec) Closure<?> config
    ) {
        report(ProblemId.of(problemName, displayName, problemGroup), config)
    }

    private final Provider<LegacyWarningReporterService> problemReporter
}
