/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2025
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant5.internal.v8.services

import groovy.transform.CompileDynamic
import groovy.util.logging.Slf4j
import org.gradle.api.Action
import org.gradle.api.Project
import org.gradle.api.internal.project.ProjectInternal
import org.ysb33r.grolifant5.api.core.ClosureUtils
import org.ysb33r.grolifant5.api.core.GrolifantProblemReporter

import javax.inject.Inject

import static org.ysb33r.grolifant5.api.core.LegacyLevel.PRE_8_11
import static org.ysb33r.grolifant5.api.core.LegacyLevel.PRE_8_8

/**
 * Implementation for the problem reporter for Gradle 9.x.
 *
 * @author Schalk W. Cronjé
 *
 * @since 5.4
 */
@CompileDynamic
@Slf4j
class GrolifantProblemReporterPre813 implements GrolifantProblemReporter {
    public static final Boolean SILENCE = System.getProperty(
        'org.ysb33r.grolifant.warnings-reporter.errors.silence',
        'false'
    ).toBoolean()

    @Inject
    GrolifantProblemReporterPre813(Project project) {
        final clazz = project.class.classLoader.loadClass('org.gradle.api.problems.Problems')
        final problems = ((ProjectInternal) project).services.get(clazz)
        this.problemReporter = PRE_8_11 ? problems.forNamespace('Grolifant') : problems.reporter
    }

    @Override
    @SuppressWarnings('CatchThrowable')
    void report(ProblemId problemId, Action<ProblemSpec> config) {
        try {
            problemReporter.reporting { spec ->
                config.execute(ProblemConverter.of(problemId, spec))
            }
        } catch (Throwable t) {
            if (!SILENCE) {
                log.info "Issue reporting warning: ${t.message}"
            }
        }
    }

    @Override
    @SuppressWarnings('CatchThrowable')
    void report(ProblemId problemId, @DelegatesTo(ProblemSpec) Closure<?> config) {
        try {
            problemReporter.reporting { spec ->
                ClosureUtils.configureItem(ProblemConverter.of(problemId, spec), config)
            }
        } catch (Throwable t) {
            if (!SILENCE) {
                log.info "Issue reporting warning: ${t.message}"
            }
        }
    }

    @Override
    void report(String problemName, String displayName, ProblemGroup problemGroup, Action<ProblemSpec> config) {
        report(ProblemId.of(problemName, displayName, problemGroup), config)
    }

    @Override
    void report(
        String problemName,
        String displayName,
        ProblemGroup problemGroup,
        @DelegatesTo(ProblemSpec) Closure<?> config
    ) {
        report(ProblemId.of(problemName, displayName, problemGroup), config)
    }

    private Object convert(ProblemId problemId) {
        org.gradle.api.problems.ProblemId.create(
            problemId.name,
            problemId.displayName,
            convert(problemId.problemGroup)
        )
    }

    private ProblemGroup convert(ProblemGroup problemGroup) {
        org.gradle.api.problems.ProblemGroup.create(problemGroup.name, problemGroup.displayName)
    }

    private final Object problemReporter

    private static class ProblemConverter implements GrolifantProblemReporter.ProblemSpec {
        final Object gps
        final ProblemId problemId

        static ProblemConverter of(ProblemId problemId1, Object gps) {
            new ProblemConverter(problemId1, gps)
        }

        ProblemConverter(ProblemId problemId1, Object gps) {
            this.gps = gps
            this.problemId = problemId1

            if (PRE_8_8) {
                this.gps.category(problemId1.name)
            } else {
                this.gps.id(problemId1.name, problemId1.displayName)
            }
        }

        @Override
        void setContextualLabel(String contextualLabel) {
            if (PRE_8_8) {
                gps.label(contextualLabel)
            } else {
                gps.contextualLabel(contextualLabel)
            }
        }

        @Override
        void setDetails(String txt) {
            gps.details(txt)
        }

        @Override
        void setDocumentedAt(String url) {
            gps.documentedAt(url)
        }

        @Override
        void setFileLocation(String path) {
            gps.fileLocation(path)
        }

        @Override
        void lineInFileLocation(String path, int line) {
            gps.lineInFileLocation(path, line)
        }

        @Override
        void lineInFileLocation(String path, int line, int column) {
            gps.lineInFileLocation(path, line, column)
        }

        @Override
        void lineInFileLocation(String path, int line, int column, int length) {
            gps.lineInFileLocation(path, line, column, length)
        }

        @Override
        void offsetInFileLocation(String path, int offset, int length) {
            gps.offsetInFileLocation(path, offset, length)
        }

        @Override
        void setSeverity(GrolifantProblemReporter.Severity severity) {
            final clazz = GrolifantProblemReporter.Severity.classLoader.loadClass(
                'org.gradle.api.problems.Severity'
            )
            gps.severity(clazz.valueOf(severity.name()))
        }

        @Override
        void setSolution(String description) {
            gps.solution(description)
        }

        @Override
        void stackLocation() {
            gps.stackLocation()
        }

        @Override
        void withException(Throwable t) {
            gps.withException(t)
        }
    }
}
