/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2025
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant5.loadable.v9

import groovy.transform.CompileStatic
import org.gradle.api.Action
import org.gradle.api.Project
import org.gradle.api.file.ArchiveOperations
import org.gradle.api.file.ConfigurableFileCollection
import org.gradle.api.file.CopySpec
import org.gradle.api.file.DeleteSpec
import org.gradle.api.file.FileSystemOperations
import org.gradle.api.file.FileTree
import org.gradle.api.model.ObjectFactory
import org.gradle.api.resources.ReadableResource
import org.gradle.api.tasks.WorkResult
import org.ysb33r.grolifant5.loadable.core.FileSystemOperationsProxy
import org.ysb33r.grolifant5.loadable.core.LoadableVersion

import javax.inject.Inject

/**
 *  Implements {@link org.ysb33r.grolifant5.api.core.FileSystemOperations} for Gradle 9.x.
 *
 * @author Schalk W. Cronjé
 *
 * @since 5.3
 */
@CompileStatic
class DefaultFileSystemOperations extends FileSystemOperationsProxy {
    @Inject
    DefaultFileSystemOperations(
        Project tempProjectReference,
        ObjectFactory objects,
        FileSystemOperations fso,
        ArchiveOperations archives
    ) {
        super(tempProjectReference, LoadableVersion.V9)
        this.fso = fso
        this.objectFactory = objects
        this.archives = archives
    }

    /**
     * Creates resource that points to a bzip2 compressed file at the given path.
     *
     * @param file File evaluated as per {@link #file}.
     * @return Readable resource
     */
    @Override
    ReadableResource bzip2Resource(Object file) {
        archives.bzip2(provideFile(file))
    }

    /**
     * Copies the specified files.
     * @param action Configures a {@link org.gradle.api.file.CopySpec}
     * @return Result of copy to check whether it was successful.
     */
    @Override
    WorkResult copy(Action<? super CopySpec> action) {
        fso.copy(action)
    }

    /**
     * Creates an empty {@link CopySpec}
     *
     * @return Empty copy specification.
     */
    @Override
    CopySpec copySpec() {
        fso.copySpec()
    }

    /**
     * Deletes the specified files.
     * @param action Configures a {@link org.gradle.api.file.DeleteSpec}
     * @return Result of deletion to check whether it was successful.
     */
    @Override
    WorkResult delete(Action<? super DeleteSpec> action) {
        fso.delete(action)
    }

    /**
     * Creates an empty file collection.
     *
     * @return Empty file collection.
     */
    @Override
    ConfigurableFileCollection emptyFileCollection() {
        objectFactory.fileCollection()
    }

    /**
     * Creates resource that points to a gzip compressed file at the given path.
     *
     * @param file File evaluated as per {@link #file}.
     * @return Readable resource
     */
    @Override
    ReadableResource gzipResource(Object file) {
        archives.gzip(provideFile(file))
    }

    /**
     * Synchronizes the contents of a destination directory with some source directories and files.
     *
     * @param action Action to configure the CopySpec.
     * @return {@link WorkResult} that can be used to check if the sync did any work.
     */
    @Override
    WorkResult sync(Action<? super CopySpec> action) {
        fso.sync(action)
    }

    /**
     * Expands a tar file into a {@link org.gradle.api.file.FileTree}.
     *
     * @param tarPath Path to tar file.
     *    Anything that can be converted with {@link #file}
     * @return Tree of tar contents.
     */
    @Override
    FileTree tarTree(Object tarPath) {
        if (tarPath instanceof ReadableResource) {
            archives.tarTree(tarPath)
        } else {
            archives.tarTree(provideFile(tarPath))
        }
    }

    /**
     * Expands a ZIP file into a {@link FileTree}.
     *
     * @param zipPath Path to tar file.
     *    Anything that can be converted with {@link #file}
     * @return Tree of ZIP contents.
     */
    @Override
    FileTree zipTree(Object zipPath) {
        if (zipPath instanceof ReadableResource) {
            archives.zipTree(zipPath)
        } else {
            archives.zipTree(provideFile(zipPath))
        }
    }

    private final FileSystemOperations fso
    private final ArchiveOperations archives
    private final ObjectFactory objectFactory
}
