/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2025
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant5.loadable.v9

import groovy.transform.CompileStatic
import org.gradle.api.Action
import org.gradle.api.Project
import org.gradle.api.artifacts.Configuration
import org.gradle.api.artifacts.ConfigurationContainer
import org.gradle.api.attributes.AttributeContainer
import org.ysb33r.grolifant5.api.core.ConfigurationPhaseOperations
import org.ysb33r.grolifant5.loadable.core.ConfigurationToolsProxy

/**
 * Provides an implementation of {@link org.ysb33r.grolifant5.api.core.ConfigurationTools} for Gradle 9.x.
 *
 * @author Schalk W. Cronjé
 *
 * @since 5.3
 */
@CompileStatic
class DefaultConfigurationTools extends ConfigurationToolsProxy {
    DefaultConfigurationTools(
        ConfigurationPhaseOperations incompleteReference, Project project) {
        super(incompleteReference, project)
        this.configurations = project.configurations
    }

    /**
     * Creates three configurations that are related to each other.
     * <p>
     * This works on the same model as to how {@code implementation}, {@code runtimeClasspath} and
     * {@code runtimeElements} are related in a JVM project.
     *
     * @param dependencyScopedConfigurationName The name of a configuration against which dependencies wil be declared.
     * @param resolvableConfigurationName The name of a configuration which can be resolved.
     * @param consumableConfigurationName The name of a configuration that can be consumed by other subprojects.
     * @param attributes Action to configure attributes for resolvable and consumable configurations.
     * @param visible Whether configurations should be marked visible or invisible.
     */
    @Override
    void createRoleFocusedConfigurations(
        String dependencyScopedConfigurationName,
        String resolvableConfigurationName,
        String consumableConfigurationName,
        boolean visible,
        Action<? super AttributeContainer> attributes
    ) {
        final depScope = configurations.dependencyScope(dependencyScopedConfigurationName)
        final resolvable = configurations.resolvable(resolvableConfigurationName)
        final consumable = configurations.consumable(consumableConfigurationName)

        [resolvable, consumable].each { c ->
            c.configure { Configuration it ->
                it.extendsFrom(depScope.get())
                it.attributes(attributes)
                it.visible = visible
            }
        }
    }

    /**
     * Creates two configurations that are related to each other and which are only meant to be used within the
     * same (sub)project.
     * <p>
     * This works on the same model as to how {@code implementation} and {@code runtimeClasspath}.
     *
     * @param dependencyScopedConfigurationName The name of a configuration against which dependencies wil be declared.
     * @param resolvableConfigurationName The name of a configuration which can be resolved.
     * @param visible Whether the configurations should be visible.
     */
    @Override
    void createLocalRoleFocusedConfiguration(
        String dependencyScopedConfigurationName,
        String resolvableConfigurationName,
        boolean visible
    ) {
        final depScope = configurations.dependencyScope(dependencyScopedConfigurationName)
        final resolvable = configurations.resolvable(resolvableConfigurationName)

        depScope.configure {
            it.visible = visible
        }
        resolvable.configure {
            it.extendsFrom(depScope.get())
            it.visible = visible
        }
    }

    /**
     * Creates a single configuration to be used for outgoing publications.
     * <p>Very useful for sharing internal outputs between subprojects.</p>
     *
     * @param configurationName Name of outgoing configuration.
     * @param visible Whether the configuration is visible.
     * @param attributes Action to configure attributes for resolvable and consumable configurations.
     */
    @Override
    void createSingleOutgoingConfiguration(
        String configurationName,
        boolean visible, Action<? super AttributeContainer> attributes
    ) {
        final consumable = configurations.consumable(configurationName)
        consumable.configure {
            it.visible = visible
            it.attributes(attributes)
        }
    }

    @Override
    protected ConfigurationContainer getConfigurations() {
        this.configurations
    }

    private final ConfigurationContainer configurations
}
