/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2025
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant5.loadable.v9

import groovy.transform.CompileStatic
import org.gradle.api.Action
import org.gradle.api.problems.ProblemGroup as GradleProblemGroup
import org.gradle.api.problems.ProblemId as GradleProblemId
import org.gradle.api.problems.ProblemReporter
import org.gradle.api.problems.ProblemSpec as GradleProblemSpec
import org.gradle.api.problems.Problems
import org.gradle.api.problems.Severity as GradleProblemSeverity
import org.ysb33r.grolifant5.api.core.ClosureUtils
import org.ysb33r.grolifant5.api.core.GrolifantProblemReporter

import javax.inject.Inject

/**
 * Implementation for the problem reporter for Gradle 9.x.
 *
 * @author Schalk W. Cronjé
 *
 * @since 5.4
 */
@CompileStatic
class DefaultGrolifantProblemReporter implements GrolifantProblemReporter {

    @Inject
    DefaultGrolifantProblemReporter(Problems problems) {
        this.problemReporter = problems.reporter
    }

    @Override
    void report(ProblemId problemId, Action<ProblemSpec> config) {
        problemReporter.report(convert(problemId)) { GradleProblemSpec spec ->
            config.execute(ProblemConverter.of(spec))
        }
    }

    @Override
    void report(ProblemId problemId, @DelegatesTo(ProblemSpec) Closure<?> config) {
        problemReporter.report(convert(problemId)) { GradleProblemSpec spec ->
            ClosureUtils.configureItem(ProblemConverter.of(spec), config)
        }
    }

    @Override
    void report(String problemName, String displayName, ProblemGroup problemGroup, Action<ProblemSpec> config) {
        report(ProblemId.of(problemName, displayName, problemGroup), config)
    }

    @Override
    void report(
        String problemName,
        String displayName,
        ProblemGroup problemGroup,
        @DelegatesTo(ProblemSpec) Closure<?> config
    ) {
        report(ProblemId.of(problemName, displayName, problemGroup), config)
    }

    private GradleProblemId convert(ProblemId problemId) {
        GradleProblemId.create(problemId.name, problemId.displayName, convert(problemId.problemGroup))
    }

    private GradleProblemGroup convert(ProblemGroup problemGroup) {
        GradleProblemGroup.create(problemGroup.name, problemGroup.displayName)
    }

    private final ProblemReporter problemReporter

    private static class ProblemConverter implements GrolifantProblemReporter.ProblemSpec {
        final GradleProblemSpec gps

        static ProblemConverter of(GradleProblemSpec gps) {
            new ProblemConverter(gps)
        }

        ProblemConverter(GradleProblemSpec gps) {
            this.gps = gps
        }

        @Override
        void setContextualLabel(String contextualLabel) {
            gps.contextualLabel(contextualLabel)
        }

        @Override
        void setDetails(String txt) {
            gps.details(txt)
        }

        @Override
        void setDocumentedAt(String url) {
            gps.documentedAt(url)
        }

        @Override
        void setFileLocation(String path) {
            gps.fileLocation(path)
        }

        @Override
        void lineInFileLocation(String path, int line) {
            gps.lineInFileLocation(path, line)
        }

        @Override
        void lineInFileLocation(String path, int line, int column) {
            gps.lineInFileLocation(path, line, column)
        }

        @Override
        void lineInFileLocation(String path, int line, int column, int length) {
            gps.lineInFileLocation(path, line, column, length)
        }

        @Override
        void offsetInFileLocation(String path, int offset, int length) {
            gps.offsetInFileLocation(path, offset, length)
        }

        @Override
        void setSeverity(GrolifantProblemReporter.Severity severity) {
            gps.severity(GradleProblemSeverity.valueOf(severity.name()))
        }

        @Override
        void setSolution(String description) {
            gps.solution(description)
        }

        @Override
        void stackLocation() {
            gps.stackLocation()
        }

        @Override
        void withException(Throwable t) {
            gps.withException(t)
        }
    }
}
