/**
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2024
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant5.api.core;

import org.gradle.api.provider.Property;
import org.gradle.api.provider.Provider;

import java.net.URI;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Callable;

/**
 * Tools for working with strings.
 *
 * @author Schalk W. Cronjé
 * @since 2.0
 */
public interface StringTools {

    /**
     * Creates a SHA-256 has of a URI.
     *
     * @param uri URI to hash
     * @return SHA-257 hash that is hex-encoded.
     */
    String hashUri(final URI uri);

    /**
     * Loads text from a resource file.
     *
     * @param resourcePath Resource path.
     *
     * @return Text
     *
     * @throws org.gradle.api.resources.ResourceException
     *
     * @since 5.0
     */
    String loadTextFromResource(String resourcePath);

    /**
     * Get final package or directory name from a URI
     *
     * @param uri URI to process
     * @return Last part of URI path.
     */
    String packageNameFromUri(final URI uri);

    /**
     * Like {@link #stringize}, but does not immediately evaluate the passed parameter.
     * The latter is only done when the provider is resolved.
     *
     * @param stringy An object that can be converted to a string or a closure that
     *                can be evaluated to something that can be converted to a string.
     * @return A provider to a string.
     *
     * @since 5.0
     */
    Provider<String> provideString(final Object stringy);

    /**
     * Like {@link #stringizeOrNull(Object)}, but does not evaluate the passed parameter immediately.
     *
     * @param stringy Anything convertible to a string.
     * @return Provider to a string. Can be empty.
     *
     * @since 5.0
     */
    Provider<String> provideStringOrNull(final Object stringy);

    /**
     * Lazy-loads text from a resource file.
     *
     * @param resourcePath Resource path.
     *
     * @return Provider to text
     *
     * @throws org.gradle.api.resources.ResourceException
     *
     * @since 5.0
     */
    Provider<String> provideTextFromResource(String resourcePath);

    /**
     * Similar to {@link #stringizeValues}, but does not evalute the parameter immediately, only when the
     * provider is resolved.
     *
     * @param props A map that is keyed on strings, but for which teh values can be converted to strings
     *              using {@link #stringize(Object)}.
     * @return A provider to a lazy-resolved map.
     *
     * @since 5.0
     */
    Provider<Map<String, String>> provideValues(Map<String, ?> props);

    /**
     * Create a URI where the user/password is masked out.
     *
     * @param uri Original URI
     * @return URI with no credentials.
     */
    URI safeUri(URI uri);

    /**
     * Converts most things to a string. Closures are evaluated as well.
     * <p>
     * Converts collections of the following recursively until it gets to a string:
     *
     * <ul>
     *   <li> {@code CharSequence} including {@code String} and {@code GString}.
     *   <li> {@link java.io.File}.
     *   <li> {@link java.nio.file.Path} is it is associated with the default provider
     *   <li> URLs and URIs.
     *   <li> Groovy Closures.
     *   <li>{@link Callable}</li>
     *   <li>{@link Optional}</li>
     *   <li>{@link Provider}</li>
     *   <li>{@link java.util.function.Supplier}</li>
     *   <li> {@link org.gradle.api.resources.TextResource}</li>
     * </ul>
     *
     * @param stringy An object that can be converted to a string or a closure that
     *                can be evaluated to something that can be converted to a string.
     * @return A string object
     */
    String stringize(final Object stringy);

    /**
     * Like {@link #stringize}, but returns {@code null} rather than throwing an exception, when item is {@code null},
     * an empty {@Link Provider} or an empty {@link java.util.Optional}.
     *
     * @param stringy
     * @return string or {@code null}
     */
    String stringizeOrNull(final Object stringy);

    /**
     * Converts a collection of most things to a list of strings. Closures are evaluated as well.
     *
     * @param stringyThings Iterable list of objects that can be converted to strings, including closure that can be evaluated
     *                      into objects that can be converted to strings.
     * @return A list of strings
     */
    List<String> stringize(final Collection<?> stringyThings);

    /**
     * Like {@link #stringize}, but drops any nulls, or empty instances of {@link Provider} and {@link Optional}.
     *
     * @param stringyThings
     * @return A list of strings
     */
    List<String> stringizeDropNull(final Collection<?> stringyThings);

    /**
     * Evaluates a map of objects to a map of strings.
     * <p>
     * Anything value that can be evaluated by {@link #stringize} is
     * evaluated
     *
     * @param props Map that will be evaluated
     * @return Converted {@code Map<String,String>}
     */
    default Map<String, String> stringizeValues(Map<String, ?> props) {
        return props.entrySet().stream().collect(
                HashMap::new,
                (map, entry) -> map.put(entry.getKey(), stringize(entry.getValue())),
                Map::putAll
        );
    }

    /**
     * Updates a {@code Property<String>}.
     *
     * THis provides a more powerful way than {@link Property#set}.
     *
     * @param provider String property
     * @param stringy Objexct that is a string or can be converted to a string.
     */
    void updateStringProperty(Property<String> provider, Object stringy);

    /**
     * Attempts to convert object to a URI.
     *
     * <p>
     * Converts collections of the following recursively until it gets to a file:
     *
     * <ul>
     *   <li> {@code CharSequence} including {@code String} and {@code GString}.
     *   <li> {@link java.io.File}.
     *   <li> {@link java.nio.file.Path} is it is associated with the default provider
     *   <li> URLs and URIs.
     *   <li> Groovy Closures.
     *   <li> {@link java.util.concurrent.Callable}.
     *   <li> {@link org.gradle.api.provider.Provider}.
     *   <li>{@link Callable}</li>
     *   <li>{@link Optional}</li>
     *   <li>{@link java.util.function.Supplier}</li>
     * </ul>
     *
     * @param uriThingy Anything that could be converted to a URI
     * @return URI object
     */
    URI urize(final Object uriThingy);
}