/**
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2024
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant5.api.core.executable;

import org.gradle.api.Action;
import org.ysb33r.grolifant5.api.core.runnable.ExecOutput;

/**
 * Describes the data stream redirections for a process including exit checks.
 * <p>
 * Similar to {@link org.gradle.process.BaseExecSpec} except to {@code getCommandLine / setCommandLine}.
 *
 * @author Schalk W. Cronjé
 * @since 2.0
 */
public interface ProcessExecutionSpec {

    /**
     * Tells whether a non-zero exit value is ignored, or an exception thrown.
     *
     * @return whether a non-zero exit value is ignored, or an exception thrown.
     */
    boolean isIgnoreExitValue();

    /**
     * Sets whether a non-zero exit value is ignored, or an exception thrown.
     *
     * @param ignoreExitValue - whether a non-zero exit value is ignored, or an exception thrown
     */
    void setIgnoreExitValue(boolean ignoreExitValue);
    /**
     * Forwards the standard output from the running process to console except if running in quiet mode.
     *
     * <p>Processes run in a worker will silently ignore this setting</p>.
     *
     * @since 5.0
     */
    void forwardOutput();

    /**
     * Forwards the error output from the running process to console except if running in quiet mode.
     *
     * <p>Error output is always written after standard output.</p>
     *
     * <p>Processes run in a worker will silently ignore this setting</p>.
     *
     * @since 5.0
     */
    void forwardErrorOutput();

    /**
     * After execution has been completed, run this function.
     * <p>
     * This will be run if there was an error during execution.
     *
     * @param resultProcessor Function to execute.
     *                        It is passed the result of the execution.
     * @since 5.0
     */
    void afterExecute(Action<ExecOutput> resultProcessor);
}
