/**
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2024
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant5.api.core.runnable;

import groovy.lang.Closure;
import org.gradle.api.Action;
import org.gradle.process.ExecResult;
import org.ysb33r.grolifant5.api.core.ClosureUtils;
import org.ysb33r.grolifant5.api.core.ProjectOperations;

/**
 * Implements the {@code exec} methods from {@link ExecMethods}, leaving it to the plugin author to
 * <p>Expects that the class implementing the interface can supply an instance of {@link ProjectOperations}</p>.
 *
 * @param <E> Execution specification which must extend {@code AbstractExecSpec}.
 *
 * @author Schalk W. Cronjé
 * @since 2.0
 */
public interface ProvisionedExecMethods<E extends AbstractExecSpec<E>> extends ExecMethods<E> {

    /**
     * Grolifant's configuration cache-safe operations.
     *
     * @return instance.
     */
    ProjectOperations getProjectOperations();

    /**
     * Create execution specification.
     *
     * @return Execution specification.
     */
    E createExecSpec();

    @Override
    default ExecResult exec(E spec) {
        return getProjectOperations().exec(execSpec -> spec.copyTo(execSpec));
    }

    @Override
    default ExecResult exec(Action<E> specConfigurator) {
        E spec = createExecSpec();
        specConfigurator.execute(spec);
        return exec(spec);
    }

    /**
     * Creates an execution specification, configures it by Groovy closure and then executes it.
     * @param specConfigurator Configurating closure.
     * @return Result of execution.
     */
    default ExecResult exec(Closure<?> specConfigurator) {
        E spec = createExecSpec();
        ClosureUtils.configureItem(spec,specConfigurator);
        return exec(spec);
    }
}
