/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2024
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant5.internal.core.runnable

import groovy.transform.CompileStatic
import org.gradle.api.Action
import org.gradle.api.Project
import org.gradle.api.logging.LogLevel
import org.ysb33r.grolifant5.api.core.executable.ProcessExecutionSpec
import org.ysb33r.grolifant5.api.core.executable.ProcessOutputProcessor
import org.ysb33r.grolifant5.api.core.runnable.ExecOutput

import javax.inject.Inject

@CompileStatic
class ProcessExecutionSpecProxy implements ProcessExecutionSpec, ProcessOutputProcessor {

    @Inject
    ProcessExecutionSpecProxy(Project tempProjectReference) {
        quietMode = tempProjectReference.logging.level == LogLevel.QUIET
    }

    @Override
    boolean isIgnoreExitValue() {
        this.ignoreExitValue
    }

    @Override
    void setIgnoreExitValue(boolean iev) {
        this.ignoreExitValue = iev
    }

    /**
     * Forwards the standard output from the running process to console.
     *
     * @since 5.0
     */
    @Override
    void forwardOutput() {
        this.forwardOutput = true
    }

    /**
     * Forwards the error output from the running process to console.
     *
     * @since 5.0
     */
    @Override
    void forwardErrorOutput() {
        this.forwardError = true
    }

    /**
     * After execution has been completed, run this function.
     * <p>
     * This will be run if there was an error during execution.
     *
     * @param resultProcessor Function to execute.
     *                        It is passed the result of the execution.
     * @since 5.0
     */
    @Override
    void afterExecute(Action<ExecOutput> resultProcessor) {
        this.actions.add(resultProcessor)
    }

    /**
     * Whether process output is forwarded to the console.
     *
     * @return {@code true} if output is forwarded.
     */
    @Override
    boolean isOutputForwarded() {
        this.forwardError && !quietMode
    }

    /**
     * Whether process error output is forwarded to the console.
     *
     * @return {@code true} if output is forwarded.
     */
    @Override
    boolean isErrorOutputForwarded() {
        this.forwardOutput && !quietMode
    }

    /**
     * Processes the output.
     *
     *
     * @param output Execution output from {@code exec()} or {@code javaexec()} calls.
     */
    @Override
    void processOutput(ExecOutput output) {
        if (outputForwarded && output.standardOutput.asText.present) {
            System.out.println(output.standardOutput.asText.get())
        }
        if (errorOutputForwarded && output.standardError.asText.present) {
            System.err.println(output.standardError.asText.get())
        }

        actions.each { it.execute(output) }
    }

    private boolean forwardOutput = false
    private boolean forwardError = false
    private boolean ignoreExitValue = false
    private final boolean quietMode
    private final List<Action<ExecOutput>> actions = []
}
