/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2024
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant5.api.core

import groovy.transform.CompileStatic
import org.ysb33r.grolifant5.api.errors.ResourceAccessException

/**
 * Helps with loading specific resources
 *
 * @author Schalk W. Cronjé
 *
 * @since 2.0
 */
@CompileStatic
class ResourceUtils {
    /**
     * Loads properties from a resource file.
     *
     * @param resourcePath Resource path.
     * @return Properties file
     */
    static Properties loadPropertiesFromResource(String resourcePath) {
        (Properties) doWithResource(resourcePath) { InputStream strm ->
            Properties props = new Properties()
            props.load(strm)
            props
        }
    }

    /**
     * Loads a resource text file into memory.
     *
     * @param resourcePath Resource path.
     * @return Contents of text file.
     */
    static String loadTextFromResource(String resourcePath) {
        doWithResource(resourcePath) { InputStream strm ->
            strm.text
        }
    }

    private static Object doWithResource(String resourcePath, Closure runner) {
        InputStream strm = ResourceUtils.classLoader.getResourceAsStream(resourcePath)
        if (!strm) {
            throw new ResourceAccessException("${resourcePath} does not exist")
        }
        strm.withCloseable runner
    }
}
