/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2025
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant5.api.core.git

import groovy.transform.CompileStatic
import org.gradle.api.Project
import org.gradle.api.file.CopySpec
import org.gradle.api.file.FileTree
import org.gradle.api.provider.Property
import org.gradle.api.provider.Provider
import org.ysb33r.grolifant5.api.core.ConfigCacheSafeOperations
import org.ysb33r.grolifant5.api.core.downloader.ArtifactDownloader
import org.ysb33r.grolifant5.api.core.downloader.ArtifactRootVerification
import org.ysb33r.grolifant5.api.core.downloader.ArtifactUnpacker
import org.ysb33r.grolifant5.api.errors.DistributionFailedException

import javax.inject.Inject

/**
 * Downloads an archive from a Git repository.
 *
 * @author Schalk W. Cronjé
 *
 * @since 2.0
 */
@CompileStatic
class GitRepoArchiveDownloader {

    /**
     * Download archive from Git Cloud provider.
     *
     * <p>Injectable constructor.</p>
     *
     * @param descriptor Cloud Git provider metadata
     * @param Project Associated project
     *
     * @since 5.4
     */
    @Inject
    GitRepoArchiveDownloader(
        final CloudGitDescriptor descriptor,
        final Project project
    ) {
        this(descriptor, ConfigCacheSafeOperations.from(project))
    }

    /**
     * Download archive from Git Cloud provider.
     *
     * @param descriptor Cloud Git provider metadata
     * @param projectOperations Project operations.
     *
     * @since 1.0.0
     */
    @SuppressWarnings('DuplicateStringLiteral')
    GitRepoArchiveDownloader(
        final CloudGitDescriptor descriptor,
        final ConfigCacheSafeOperations gtc
    ) {
        this.ccso = gtc

        final gitName = "${descriptor.name.toLowerCase()}-cache"
        final orgP = descriptor.organisationProvider
        final repoP = descriptor.repositoryProvider
        final uriP = descriptor.archiveUriProvider
        final offP = ccso.projectTools().offlineProvider

        this.downloadRoot = ccso.providerTools().property(File).convention(gtc.fsOperations().gradleUserHomeDir)

        final safeUriProvider = descriptor.archiveUriProvider.map {
            ccso.stringTools().safeUri(it).toString()
        }

        final unpacker = new ArtifactUnpacker() {
            @Override
            void accept(File source, File destDir) {
                final FileTree archiveTree = gtc.fsOperations().zipTree(source)
                gtc.fsOperations().copy { CopySpec cs ->
                    cs.from archiveTree
                    cs.into destDir
                }
            }
        }
        final verifyRoot = new ArtifactRootVerification() {
            @Override
            File apply(File unpackedRoot) {
                List<File> dirs = gtc.fsOperations().listDirs(unpackedRoot)
                if (dirs.empty) {
                    throw new DistributionFailedException(
                        "Download for '${safeUriProvider.get()}' does not contain any directories. " +
                            'Expected to find exactly 1 directory.'
                    )
                }
                if (dirs.size() != 1) {
                    throw new DistributionFailedException(
                        "Download for '${safeUriProvider.get()}' contains too many directories. " +
                            'Expected to find exactly 1 directory.'
                    )
                }
                dirs[0]
            }
        }

        final basePathProvider = orgP.zip(repoP) { org, repo ->
            ccso.fsOperations().toSafeFile(gitName, org.toLowerCase(Locale.US), repo.toLowerCase(Locale.US)).path
        }

        final distNameProvider = orgP.zip(repoP) { org, repo ->
            "${gitName}:${org}/${repo}"
        }

        this.downloadProvider = ccso.providerTools().provider { ->
            new Tuple6(
                uriP.get(),
                basePathProvider.get(),
                downloadRoot.get(),
                offP.get(),
                safeUriProvider.get(),
                distNameProvider.get()
            )
        }.map {
            new ArtifactDownloader(
                (URI) it.v1,
                (File) it.v3,
                ccso,
                (String) it.v2,
                verifyRoot,
                unpacker,
                null
            ).getFromCache((String) it.v5, (Boolean) it.v4, ccso.execTools().downloader((String) it.v6))
        }
    }

    /**
     * Returns the location which is the top or home folder for a distribution.
     *
     *  This value is affected by {@link #setDownloadRoot(java.io.File)} and
     *  the parameters passed in during construction time.
     *
     * @return Location of the distribution.
     *
     * @deprecated
     */
    @Deprecated
    File getArchiveRoot() {
        downloadProvider.get()
    }

    /**
     * Returns the location which is the top or home folder for a distribution.
     *
     *  This value is affected by {@link #setDownloadRoot(java.io.File)} and
     *  the parameters passed in during construction time.
     *
     * @return Location of the distribution.
     *
     * @since 5.4
     */
    Provider<File> getArchiveRootProvider() {
        this.downloadProvider
    }

    /**
     * Sets a download root directory for the distribution.
     *
     * If not supplied the default is to use the Gradle User Home.
     * This method is provided for convenience and is mostly only used for testing
     * purposes.
     *
     * The folder will be created at download time if it does not exist.
     *
     * @param downloadRootDir Any writeable directory on the filesystem.
     */
    void setDownloadRoot(File downloadRootDir) {
        this.downloadRoot.set(downloadRootDir)
    }

    private final ConfigCacheSafeOperations ccso
    private final Property<File> downloadRoot
    private final Provider<File> downloadProvider
}
