/**
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2025
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant5.api.core.jvm;

import org.gradle.api.provider.Provider;
import org.gradle.jvm.toolchain.JavaLauncher;

import java.util.Map;

/**
 * An interface to set the most import Java fork options without relying on the actual Gradle API
 * {@link org.gradle.process.JavaForkOptions}.
 *
 * @author Schalk W. Cronjé
 * @since 5.4
 */
public interface GrolifantSimpleSetJavaForkOptions {
    /**
     * Sets an alternative Java launcher.
     *
     * @param launcher Provider to a launcher.
     */
    void setJavaLauncher(Provider<JavaLauncher> launcher);

    /**
     * Sets the system properties to use. Null values will be converted to empty strings.
     *
     * <p>This will add to the system properties of the Java process.</p>
     * @param sysProps System properties that will be lazy-evaluated.
     */
    void setSystemProperties(Map<String,?> sysProps);

    /**
     * Sets the system properties to use. Null values will be converted to empty strings.
     *
     * <p>This will add to the system properties of the Java process.</p>
     *
     * @param sysProps Provider to a map of system properties.
     */
    void setSystemProperties(Provider<Map<String, String>> sysProps);

    /**
     * Set the environment from a map. Null values will be dropped.
     *
     * <p>This will be added to the environment of the actual Java process.</p>
     *
     * @param env Environment that will be lazy-evaluated.
     */
    void setEnvironment(Map<String, ?> env);

    /**
     * Set the environment from a provided map.
     *
     * <p>This will be added to the environment of the actual Java process.</p>
     *
     * @param env Environment that will be lazy-evaluated.
     */
    void setEnvironment(Provider<Map<String, String>> env);

    /**
     * Sets the JVM args to use. Null values will be dropped.
     *
     * <p>This will add to the JVM arguments of the Java process.</p>
     *
     * @param args List of arguments. WIll be lazy-evaluated.
     */
    void setJvmArgs(Iterable<?> args);

    /**
     * Sets the JVM args to use.
     *
     * <p>This will add to the JVM arguments of the Java process.</p>
     *
     * @param args Provider to a list of arguments.
     */
    void setJvmArgs(Provider<Iterable<String>> args);

    /**
     * Sets the maximum heap size.
     *
     * @param size Heap size.
     */
    void setMaxHeapSize(String size);

    /**
     * Sets the maximum heap size.
     *
     * @param size Provider to a heap size.
     */
    void setMaxHeapSize(Provider<String> size);

    /**
     * Sets the minimum heap size.
     *
     * @param size Heap size.
     */
    void setMinHeapSize(String size);

    /**
     * Sets the minimum heap size.
     *
     * @param size Provider to a heap size.
     */
    void setMinHeapSize(Provider<String> size);
}
