/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2025
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant5.api.core.runnable

import groovy.transform.CompileStatic
import org.gradle.api.provider.ListProperty
import org.gradle.api.provider.Provider
import org.ysb33r.grolifant5.api.core.AllArgsProvider
import org.ysb33r.grolifant5.api.core.CmdlineArgumentSpec
import org.ysb33r.grolifant5.api.core.ProjectOperations
import org.ysb33r.grolifant5.api.core.ProviderTools
import org.ysb33r.grolifant5.api.core.StringTools

/**
 * Abstract class to set command-line parameters.
 *
 * @author Schalk W. Cronjé
 *
 * @since 2.0
 */
@CompileStatic
class AbstractCmdlineArgumentSpec implements CmdlineArgumentSpec, AllArgsProvider {

    /**
     * Return list of arguments to the entrypoint.
     *
     * @return List of resolved arguments.
     */
    @Override
    List<String> getArgs() {
        this.arguments.get()
    }

    /**
     * Add arguments to the entrypoint.
     *
     * @param args Any arguments resolvable to strings.
     */
    @Override
    void args(Object... args) {
        this.arguments.addAll(stringTools.provideStringsDropNull(args.toList()))
    }

    /**
     * Add arguments to the entrypoint.
     *
     * @param args Any arguments resolvable to strings.
     */
    @Override
    void args(Iterable<?> args) {
        this.arguments.addAll(stringTools.provideStringsDropNull(args.toList()))
    }

    /**
     * Replace current arguments with a new set.
     *
     * @param args Any arguments resolvable to strings.
     */
    @Override
    void setArgs(Iterable<?> args) {
        this.arguments.set([])
        this.arguments.addAll(stringTools.provideStringsDropNull(args.toList()))
    }

    /**
     * Replace current arguments with a new set.
     *
     * @param args Any arguments resolvable to strings.
     */
    void setArgs(List<String> args) {
        this.arguments.set([])
        this.arguments.addAll(args)
    }

    /**
     * A provider to arguments that will be inserted before any supplied arguments.
     *
     * @return Arguments provider
     */
    @Override
    Provider<List<String>> getPreArgs() {
        this.preArgsProvider
    }

    /**
     * All defined arguments, plus all arguments providers via the command-line argument providers.
     *
     * @return Provider to a fully-resolved set of arguments.
     */
    @Override
    Provider<List<String>> getAllArgs() {
        this.allArgsProvider
    }

    /**
     * Add lazy-evaluated providers of arguments.
     *
     * @param providers One or more providers or string lists.
     */
    @Override
    void addCommandLineArgumentProviders(Provider<List<String>>... providers) {
        this.externalArgs.addAll(providers)
    }

    /**
     * Get current list of command-line argument providers.
     *
     * @return Current list of providers.
     */
    @Override
    List<Provider<List<String>>> getCommandLineArgumentProviders() {
        externalArgs
    }

//    // Will be removed in 6.0
//    @Deprecated
//    protected AbstractCmdlineArgumentSpec(
//        StringTools stringTools,
//        ProviderFactory providers
//    ) {
//        this.stringTools = stringTools
//        this.allArgsProvider = providers.provider(createAllArgsProvider(this))
//        this.preArgsProvider = providers.provider { -> Collections.EMPTY_LIST }
//    }

    protected AbstractCmdlineArgumentSpec(
        StringTools stringTools,
        ProviderTools providers
    ) {
        this.stringTools = stringTools
        this.providerTools = providers
        this.arguments = providers.listProperty(String).convention(Collections.emptyList())
        this.preArgsProvider = providers.provider { -> Collections.EMPTY_LIST }
        this.allArgsProvider = createAllArgsProvider()
    }

    // Will be removed in 6.0
    @Deprecated
    protected AbstractCmdlineArgumentSpec(ProjectOperations po) {
        this(po.stringTools, po.providerTools)
    }

    @SuppressWarnings('UnnecessaryGetter')
    private Provider<List<String>> createAllArgsProvider() {
        this.arguments.map { a ->
            List<String> flattenedArguments = []
            flattenedArguments.addAll(preArgs.getOrElse(Collections.EMPTY_LIST))
            flattenedArguments.addAll(a)
            commandLineArgumentProviders.each { prv ->
                flattenedArguments.addAll(prv.getOrElse(Collections.EMPTY_LIST))
            }
            flattenedArguments.flatten() as List<String>
        }
    }

    private final StringTools stringTools
    private final ProviderTools providerTools
    private final ListProperty<String> arguments
    private final List<Provider<List<String>>> externalArgs = []
    private final Provider<List<String>> preArgsProvider
    private final Provider<List<String>> allArgsProvider
}
