/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2025
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant5.api.core.runnable

import groovy.transform.CompileStatic
import org.gradle.api.Project
import org.gradle.api.provider.ListProperty
import org.gradle.api.provider.Property
import org.gradle.api.provider.Provider
import org.ysb33r.grolifant5.api.core.ConfigCacheSafeOperations
import org.ysb33r.grolifant5.api.core.downloader.ExecutableDownloader
import org.ysb33r.grolifant5.api.errors.ConfigurationException
import org.ysb33r.grolifant5.api.errors.ExecConfigurationException
import org.ysb33r.grolifant5.internal.core.runnable.RunnableUtils

import java.util.concurrent.Callable

import static org.ysb33r.grolifant5.internal.core.runnable.RunnableUtils.initWindowsSearchOrder

/**
 * A base class for implementing toolchains based up on external tools that can be download or found installed on the
 * local filesystem.
 *
 * @author Schalk W. Cronjé
 *
 * @since 5.5
 */
@CompileStatic
abstract class AbstractNativeToolchain implements ToolLocation {

    @Override
    void executableByVersion(Object version) {
        unsetPaths()
        ccso.stringTools().updateStringProperty(this.version, version)
    }

    @Override
    void executableByPath(Object path) {
        unsetNonPaths()
        ccso.fsOperations().updateFileProperty(this.path, path)
    }

    @Override
    void executableBySearchPath(Object baseName) {
        unsetNonPaths()
        ccso.stringTools().updateStringProperty(this.baseNameForPathSearch, baseName)
    }

    @Override
    void setWindowsExtensionSearchOrder(Iterable<String> order) {
        this.windowSearchOrder.value(order)
    }

    @Override
    void setWindowsExtensionSearchOrder(String... order) {
        this.windowSearchOrder.value(order.toList())
    }

    @Override
    List<String> getWindowsExtensionSearchOrder() {
        this.windowSearchOrder.getOrNull()
    }

    @Override
    Provider<File> getExecutable() {
        this.executable
    }

    Provider<String> getExecutableVersion() {
        this.executableVersion
    }

    protected AbstractNativeToolchain(Project project) {
        this.ccso = ConfigCacheSafeOperations.from(project)
        this.version = project.objects.property(String)
        this.path = project.objects.property(File)
        this.baseNameForPathSearch = project.objects.property(String)
        this.windowSearchOrder = project.objects.listProperty(String).value(initWindowsSearchOrder(project))

        final resolveFromVersion = this.version.map {
            if (owner.downloader) {
                downloader.getByVersion(it)
            } else {
                throw new ExecConfigurationException('A version was defined, but no downloader was configured')
            }
        }

        final resolveFromSearchPath = this.baseNameForPathSearch.map {
            RunnableUtils.findExecutableOnPath(it, owner.windowSearchOrder.getOrNull())
        }

        final resolveFailed = project.provider({ ->
            throw new ExecConfigurationException('No version, location, or search path has been configured')
        } as Callable<File>)

        this.executable = project.objects.property(File).convention(
            resolveFromVersion.orElse(this.path.orElse(resolveFromSearchPath.orElse(resolveFailed)))
        )
        this.executable.disallowUnsafeRead()

        this.executableVersion = project.objects.property(String).convention(
            this.version.orElse(this.executable.map { owner.runExecutableAndReturnVersion(it) })
        )
        this.executableVersion.disallowUnsafeRead()
    }

    protected final ConfigCacheSafeOperations ccso

    /**
     * Gets the downloader implementation.
     *
     * Can throw an exception if downloading is not supported.
     *
     * @return A downloader that can be used to retrieve an executable.
     *
     */
    abstract protected ExecutableDownloader getDownloader()

    /**
     * Runs the executable and returns the version.
     *
     * See {@link org.ysb33r.grolifant5.api.core.ExecTools#parseVersionFromOutput} as a helper to implement this method.
     *
     * @param exe Executable location
     * @return Version string.
     * @throws org.ysb33r.grolifant5.api.errors.ConfigurationException if configuration is not correct or version could
     *   not be determined.
     */
    abstract protected String runExecutableAndReturnVersion(File exe) throws ConfigurationException

    /**
     * Unset both the fixed path and search path.
     *
     * THis can be used by derived classes to ensure that paths are not included in exectuble resolution
     */
    protected void unsetPaths() {
        this.path.value((File) null)
        this.baseNameForPathSearch.value((String) null)
    }

    /**
     * This is used when path or search path is set to unset other search mechanisms.
     *
     * This is primarily intended to be used within Grolifant itself as part of its base classes and MUST be overridden
     * in those.
     *
     * The default operation is NOOP.
     */
    protected void unsetNonPaths() {
        this.version.value((String) null)
    }

    private final Property<String> version
    private final Property<File> path
    private final Property<String> baseNameForPathSearch
    private final ListProperty<String> windowSearchOrder
    private final Property<File> executable
    private final Property<String> executableVersion
}
