/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2025
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant5.internal.core.runnable

import groovy.transform.CompileStatic
import org.gradle.api.Project
import org.gradle.api.provider.Provider
import org.ysb33r.grolifant5.api.core.OperatingSystem
import org.ysb33r.grolifant5.api.core.Transform
import org.ysb33r.grolifant5.api.errors.ExecutionException

import java.util.function.Function
import java.util.regex.Pattern

import static org.ysb33r.grolifant5.api.core.StringTools.EMPTY

/**
 * Simple utilities to be used internally for ruunables and executables.
 *
 * @author Schalk W. Cronjé
 *
 * @since 5.5
 */
@CompileStatic
class RunnableUtils {
    public static final OperatingSystem OS = OperatingSystem.current()

    /**
     * An initial search path for windows.
     *
     * @param project Contextual project
     * @return Provider to a list of search paths. Can be empty or not present, but not {@code null}.
     */
    static Provider<List<String>> initWindowsSearchOrder(Project project) {
        if (OS.windows) {
            project.providers.environmentVariable('PATHEXT')
                .orElse(EMPTY)
                .map { path ->
                    final List<String> entries = []
                    if (path) {
                        for (String entry : path.split(Pattern.quote(OS.pathSeparator))) {
                            entries.add(entry)
                        }
                    } else {
                        entries.addAll('.COM', '.EXE', '.BAT', '.CMD')
                    }
                    entries
                } as Provider<List<String>>
        } else {
            project.provider { -> (List) null }
        }
    }

    /**
     * Searches for an executable on the path.
     *
     * @param baseName Basename of executable.
     * @param extSearchOrder Extension search order. Ignored if not Windows.
     * @return Locastion of executable.
     *
     * @throw ExecutionException if executable not found
     */
    static File findExecutableOnPath(String baseName, List<String> extSearchOrder) {
        final found = OS.windows ? findExecutableOnPathWindows(baseName, extSearchOrder) :
            findExecutableOnPathNonWindows(baseName)

        if (found == null) {
            throw new ExecutionException("Could not locate ${baseName} in search path")
        }

        found
    }

    private static File findExecutableOnPathWindows(String baseName, List<String> extSearchOrder) {
        final targets = Transform.toList(
            extSearchOrder,
            { "${baseName}${it}".toString() } as Function<String, String>
        )
        for (String target : targets) {
            final found = OperatingSystem.current().findInPath(target)
            if (found) {
                return found
            }
        }
        null
    }

    private static File findExecutableOnPathNonWindows(String baseName) {
        OperatingSystem.current().findInPath(baseName)
    }
}
