/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2025
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant5.internal.core.services

import groovy.transform.CompileStatic
import groovy.transform.Synchronized
import groovy.util.logging.Slf4j
import org.gradle.api.Project
import org.gradle.api.provider.Provider
import org.gradle.api.services.BuildService
import org.gradle.api.services.BuildServiceSpec
import org.ysb33r.grolifant5.api.core.GrolifantProblemReporter.ProblemId
import org.ysb33r.grolifant5.api.core.GrolifantProblemReporter.ProblemSpec
import org.ysb33r.grolifant5.api.core.LegacyLevel
import org.ysb33r.grolifant5.api.core.StringTools
import org.ysb33r.grolifant5.api.errors.UnsupportedConfigurationException

/**
 * Handles a warning report prior to Groovy 8.6.
 *
 * @author Schalk W. Cronjé
 *
 * @since 5.4
 */
@CompileStatic
@SuppressWarnings(['AbstractClassWithoutAbstractMethod', 'AbstractClassWithPublicConstructor'])
@Slf4j
abstract class LegacyWarningReporterService implements
    BuildService<LegacyWarningReporterServiceParameters>, AutoCloseable {

    public static final String LEGACY_WARNING_REPORTER_SERVICE = 'grolifant-legacy-warning-reporter-service'
    public static final boolean USE_LEGACY_WARNING_REPORTER = LegacyLevel.PRE_8_11

    static Provider<LegacyWarningReporterService> registerIfAbsent(Project project) {
        if (USE_LEGACY_WARNING_REPORTER) {
            final dest = project.rootProject.layout.buildDirectory.file('reports/problems/problems-report.txt')
            project.gradle.sharedServices.registerIfAbsent(
                LEGACY_WARNING_REPORTER_SERVICE,
                LegacyWarningReporterService
            ) { BuildServiceSpec<LegacyWarningReporterServiceParameters> spec ->
                spec.parameters.reportFile.set(dest)
            }
        } else {
            throw new UnsupportedConfigurationException('This service can only be used on Gradle versions < 8.11')
        }
    }

    LegacyWarningReporterService() {
        this.reportFile = parameters.reportFile.get().asFile
        this.reportFile.parentFile.mkdirs()
        this.reportFile.delete()
    }

    @Synchronized
    void report(ProblemId id, ProblemSpec spec) {
        final LegacyProblemDetails details = (LegacyProblemDetails) spec
        final dIndent = StringTools.SPACE * 2
        reportFile.withWriterAppend { w ->
            w.println "Id: ${id.name}"
            w.println "Description: ${id.displayName}"
            w.println "Group: ${id.problemGroup.name}"
            w.println "  ${id.problemGroup.displayName}"
            w.println 'Details:'
            details.optContextualLabel.ifPresent { w.println "${dIndent}Label: ${it}" }
            details.optDetails.ifPresent { w.println "${dIndent}Problem: ${it}" }
            details.optSeverity.ifPresent { w.println "${dIndent}Severity: ${it}" }
            details.optFileLocation.ifPresent { w.println "${dIndent}Location: ${it}" }
            details.optDocumentedAt.ifPresent { w.println "${dIndent}Documented at: ${it}" }
            details.optSolution.ifPresent { w.println "${dIndent}Solution: ${it}" }
            details.optException.ifPresent {
                w.println "${dIndent}Exception: ${it.class.canonicalName}"
                w.println "${dIndent}           ${it.message}"
            }
            w.println(END_SECTION * 80)
        }
    }

    @Override
    void close() throws Exception {
        if (reportFile.exists() && reportFile.size()) {
            final path = reportFile.toURI().toString()
            log.info("\n\n[Grolifant Backport] Problems report is available at: ${path}\n\n")
        }
    }

    private final File reportFile
    private static final String END_SECTION = '-'
}
