/**
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2025
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant5.api.core;

import groovy.lang.Closure;
import groovy.lang.DelegatesTo;
import org.gradle.api.Action;

import java.util.Locale;

/**
 * The Grolifant Problem Reporter.
 *
 * @author Schalk W. Cronjé
 * @since 5.4
 */
public interface GrolifantProblemReporter {
    /**
     * Defines a problem group.
     */
    static class ProblemGroup {
        final String name;
        final String displayName;

        public String getDisplayName() {
            return displayName;
        }

        public String getName() {
            return name;
        }

        public static ProblemGroup of(String name, String displayName) {
            return new ProblemGroup(name, displayName);
        }

        ProblemGroup(String name, String displayName) {
            this.name = name;
            this.displayName = displayName;
        }
    }

    /**
     * Defines a problem id.
     */
    static class ProblemId {
        final String name;
        final String displayName;
        final ProblemGroup problemGroup;

        public ProblemGroup getProblemGroup() {
            return problemGroup;
        }

        public String getDisplayName() {
            return displayName;
        }

        public String getName() {
            return name;
        }

        public static ProblemId of(String name, String displayName, ProblemGroup problemGroup) {
            return new ProblemId(name, displayName, problemGroup);
        }

        public static ProblemId of(String name, String displayName, String groupName, String groupDisplayName) {
            return new ProblemId(name, displayName, groupName, groupDisplayName);
        }

        ProblemId(String name, String displayName, String groupName, String groupDisplayName) {
            this(name, displayName, ProblemGroup.of(groupName, groupDisplayName));
        }

        ProblemId(String name, String displayName, ProblemGroup problemGroup) {
            this.name = name;
            this.displayName = displayName;
            this.problemGroup = problemGroup;
        }
    }

    /**
     * Severity.
     */
    static enum Severity {
        ADVICE,
        ERROR,
        WARNING
    }

    /**
     * A configurable problem specification.
     *
     * <p>Currently, it does not support additional data akin to what Gradle 8.13+ offers.</p>
     */
    static interface ProblemSpec {

        /**
         * Declares a short, but context-dependent message for this problem.
         *
         * @param contextualLabel Contextual message.
         */
        void setContextualLabel(String contextualLabel);

        /**
         * Declares a long description detailing the problem.
         *
         * @param txt Details.
         */
        void setDetails(String txt);

        /**
         * Declares where this problem is documented.
         *
         * @param url Documentation url.
         */
        void setDocumentedAt(String url);

        /**
         * Declares that this problem is in a file.
         *
         * @param path File path
         */
        void setFileLocation(String path);

        /**
         * Declares that this problem is in a file on a line.
         *
         * @param path File path
         * @param line Line number
         */
        void lineInFileLocation(String path, int line);

        /**
         * Declares that this problem is in a file with on a line at a certain position.
         *
         * @param path File path
         * @param line Line number
         * @param column Column start
         */
        void lineInFileLocation(String path, int line, int column);

        /**
         * Declares that this problem is in a file with on a line at a certain position.
         *
         * @param path File path
         * @param line Line number
         * @param column Column start
         * @param length Length of text
         */
        void lineInFileLocation(String path, int line, int column, int length);

        /**
         * Declares that this problem is in a file at a certain global position with a given length.
         *
         * @param path File path
         * @param offset File offset
         * @param length Problem length
         */
        void offsetInFileLocation(String path, int offset, int length);

        /**
         * Declares the severity of the problem.
         *
         * @param severity Severity
         */
        void setSeverity(Severity severity);

        /**
         * Declares the severity of the problem.
         *
         * @param severity Severity
         */
        default void setSeverity(String severity) {
            setSeverity(Severity.valueOf(severity.toUpperCase(Locale.US)));
        }

        /**
         * Declares solutions and advice that contain context-sensitive data.
         *
         * @param description Solution
         */
        void setSolution(String description);

        /**
         * Declares that this problem is at the same place where it's reported.
         *
         * <p>Could be a no NOOP on older versoins of Gradle.</p>
         */
        void stackLocation();

        /**
         * Declares the exception causing this problem.
         *
         * @param t Exception
         */
        void withException(Throwable t);
    }

    /**
     * Report a problem with additional information.
     *
     * @param problemId Problem identifier.
     * @param config Configurator for a supplied problem specification
     */
    void report(ProblemId problemId, Action<ProblemSpec> config);

    /**
     * Report a problem with additional information.
     *
     * @param problemId Problem identifier.
     * @param config Configurator for a supplied problem specification
     */
    void report(ProblemId problemId, @DelegatesTo(ProblemSpec.class) Closure<?> config);

    /**
     * Report a problem with additional information.
     *
     * @param problemName Problem name.
     * @param displayName Problem display name.
     * @param problemGroup Problem group.
     * @param config Configurator for a supplied problem specification
     */
    void report(String problemName, String displayName, ProblemGroup problemGroup, Action<ProblemSpec> config);

    /**
     * Report a problem with additional information.
     *
     * @param problemName Problem name.
     * @param displayName Problem display name.
     * @param pgName Problem group.
     * @param pgDisplay Problem group display.
     * @param config Configurator for a supplied problem specification
     */
    default void report(
            String problemName,
            String displayName,
            String pgName,
            String pgDisplay ,
            Action<ProblemSpec> config
    ) {
        report(problemName, displayName, ProblemGroup.of(pgName, pgDisplay), config);
    }

    /**
     * Report a problem with additional information.
     *
     * @param problemName Problem name.
     * @param displayName Problem display name.
     * @param problemGroup Problem group.
     * @param config Configurator for a supplied problem specification
     */
    void report(
            String problemName,
            String displayName,
            ProblemGroup problemGroup,
            @DelegatesTo(ProblemSpec.class) Closure<?> config
    );

    /**
     * Report a problem with additional information.
     *
     * @param problemName Problem name.
     * @param displayName Problem display name.
     * @param pgName Problem group.
     * @param pgDisplay Problem group display.
     * @param config Configurator for a supplied problem specification
     */
    default void report(
            String problemName,
            String displayName,
            String pgName,
            String pgDisplay ,
            @DelegatesTo(ProblemSpec.class) Closure<?> config
    ) {
        report(problemName, displayName, ProblemGroup.of(pgName, pgDisplay), config);
    }

}
