/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2025
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant5.api.core.git

import groovy.transform.CompileStatic
import org.gradle.api.provider.Property
import org.gradle.api.provider.Provider
import org.ysb33r.grolifant5.api.core.ConfigCacheSafeOperations
import org.ysb33r.grolifant5.api.core.StringTools

/**
 * Base class for implementing common functionality for Gitlab & GitHub.
 *
 * @author Schalk W. Cronjé
 *
 * @since 2.0
 */
@CompileStatic
abstract class AbstractCloudGit implements CloudGitConfigurator {

    final String name
    final Provider<URI> archiveUriProvider

    @Override
    void setBaseUri(URI uri) {
        this.baseUri.set(uri)
    }

    /**
     * Organisation on Git cloud service
     *
     * @return Organisation
     *
     * @since 5.4
     */
    @Override
    Provider<String> getOrganisationProvider() {
        this.org
    }

    /** Set the organisation.
     *
     * @param org Anything convertible using {@link StringTools#stringize}
     */
    @Override
    void setOrganisation(Object org) {
        stringTools.updateStringProperty(this.org, org)
    }

    /**
     * Repository on Git cloud service
     *
     * @return Repository
     *
     * @since 5.4
     */
    @Override
    Provider<String> getRepositoryProvider() {
        this.repo
    }

    /** Set the repository
     *
     * @param repo Anything convertible using {@link StringTools#stringize}
     */
    @Override
    void setRepository(Object repo) {
        stringTools.updateStringProperty(this.repo, repo)
    }

    /** Set the branch.
     *
     * Overrides {@link #setTag} and {@link #setCommit}.
     *
     * @param branch Anything convertible using {@link StringTools#stringize}
     */
    @Override
    void setBranch(Object branch) {
        stringTools.updateStringProperty(this.identifier, branch)
    }

    /** Set the tag.
     *
     * Overrides {@link #setBranch} and {@link #setCommit}.
     *
     * @param tag Anything convertible using {@link StringTools#stringize}
     */
    @Override
    void setTag(Object tag) {
        stringTools.updateStringProperty(this.identifier, tag)
    }

    /** Set the commit to use.
     *
     * Overrides {@link #setTag} and {@link #setBranch}.
     *
     * @param commit Anything convertible using {@link StringTools#stringize}
     */
    @Override
    void setCommit(Object commit) {
        stringTools.updateStringProperty(this.identifier, commit)
    }

    /** Content as a string
     *
     * @return Equivalent of {@link #getArchiveUri}.
     */
    @Override
    String toString() {
        archiveUriProvider.get()
    }

    /** Constructs a representation of a cloud Git provider
     *
     * @param name Name of provider
     * @param baseUri Base URI to access provider
     */
    protected AbstractCloudGit(final ConfigCacheSafeOperations po, final String name, String baseUri) {
        this.name = name
        this.stringTools = po.stringTools()
        this.baseUri = po.providerTools().property(URI).convention(baseUri.toURI())
        this.org = po.providerTools().property(String)
        this.repo = po.providerTools().property(String)
        this.identifier = po.providerTools().property(String)

        this.archiveUriProvider = this.baseUri.map {
            new URI(
                it.scheme,
                it.rawAuthority,
                it.path.empty ? "/${archivePath}" : "${it.path}/${archivePath}",
                null,
                null
            )
        }
    }

    /** Returns an identifier of an instance in the repository.
     *
     * @return A branch, tag or commit.
     *
     * @deprecated
     */
    @Deprecated
    protected String getIdentifier() {
        identifierProvider.get()
    }

    /** Returns an identifier of an instance in the repository.
     *
     * @return A branch, tag or commit.
     */
    protected Provider<String> getIdentifierProvider() {
        this.identifier
    }

    /** Calculates an archive path for the specific repository type.
     *
     * @return Returns a path that can be used to locate the archive.
     *   This path relative to the base URI.
     */
    abstract protected String getArchivePath()

    private final Property<URI> baseUri
    private final Property<String> identifier
    private final Property<String> org
    private final Property<String> repo
    private final StringTools stringTools
}
