/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2025
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant5.internal.core

import groovy.transform.CompileStatic
import org.gradle.api.Project
import org.gradle.api.provider.MapProperty
import org.gradle.api.provider.Property
import org.gradle.api.provider.Provider
import org.gradle.process.ProcessForkOptions
import org.ysb33r.grolifant5.api.core.ConfigCacheSafeOperations
import org.ysb33r.grolifant5.api.core.FileSystemOperations
import org.ysb33r.grolifant5.api.core.GrolifantProcessForkOptions
import org.ysb33r.grolifant5.api.core.ProviderTools
import org.ysb33r.grolifant5.api.core.StringTools

import javax.inject.Inject

import static org.ysb33r.grolifant5.api.core.StringTools.EMPTY

/**
 * Implements a {@link GrolifantProcessForkOptions} that can be easily reused
 *
 * @author Schalk W. Cronjé
 *
 * @since 5.5
 */
@CompileStatic
class SimpleGrolifantProcessForkOptions implements GrolifantProcessForkOptions {
    @Inject
    SimpleGrolifantProcessForkOptions(Project project) {
        final ccso = ConfigCacheSafeOperations.from(project)
        this.fsOperations = ccso.fsOperations()
        this.stringTools = ccso.stringTools()
        this.providerTools = ccso.providerTools()
        this.exe = project.objects.property(File)
        this.exeAsString = this.exe.map { it.absolutePath }
        this.workdir = project.objects.property(File)
        this.env = project.objects.mapProperty(String, String).convention([:])
    }

    @Override
    String getExecutable() {
        this.exeAsString.get()
    }

    @Override
    void setExecutable(String var1) {
        fsOperations.updateFileProperty(this.exe, var1)
    }

    @Override
    void setExecutable(Object var1) {
        fsOperations.updateFileProperty(this.exe, var1)
    }

    @Override
    GrolifantProcessForkOptions executable(Object var1) {
        fsOperations.updateFileProperty(this.exe, var1)
        this
    }

    @Override
    File getWorkingDir() {
        this.workdir.get()
    }

    @Override
    void setWorkingDir(File var1) {
        fsOperations.updateFileProperty(this.workdir, var1)
    }

    @Override
    void setWorkingDir(Object var1) {
        fsOperations.updateFileProperty(this.workdir, var1)
    }

    @Override
    GrolifantProcessForkOptions workingDir(Object var1) {
        fsOperations.updateFileProperty(this.workdir, var1)
        this
    }

    @Override
    Map<String, Object> getEnvironment() {
        this.env.get() as Map<String, Object>
    }

    @Override
    void setEnvironment(Map<String, ?> var1) {
        this.env.set([:])
        this.env.putAll(stringTools.provideValuesKeepNull(var1))
    }

    @Override
    GrolifantProcessForkOptions environment(Map<String, ?> var1) {
        this.env.putAll(stringTools.provideValuesKeepNull(var1))
        this
    }

    @Override
    GrolifantProcessForkOptions environment(String var1, Object var2) {
        this.env.put(var1, providerTools.provider { -> stringTools.stringizeOrNull(var2) ?: EMPTY })
        this
    }

    @Override
    GrolifantProcessForkOptions copyTo(ProcessForkOptions var1) {
        if (this.exe.present) {
            var1.executable = this.executable
        }

        if (this.workdir.present) {
            var1.workingDir = this.workingDir
        }

        var1.environment = this.env.get()
        this
    }

    private final Property<File> exe
    private final Property<File> workdir
    private final Provider<String> exeAsString
    private final MapProperty<String, String> env
    private final ProviderTools providerTools
    private final StringTools stringTools
    private final FileSystemOperations fsOperations
}
