/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2025
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant5.internal.core.jvm

import groovy.transform.CompileStatic
import org.gradle.api.Project
import org.gradle.api.file.RegularFileProperty
import org.gradle.api.provider.ListProperty
import org.gradle.api.provider.MapProperty
import org.gradle.api.provider.Property
import org.gradle.api.provider.Provider
import org.gradle.jvm.toolchain.JavaLauncher
import org.ysb33r.grolifant5.api.core.ConfigCacheSafeOperations
import org.ysb33r.grolifant5.api.core.jvm.GrolifantSimpleJavaForkOptions

import static java.util.Collections.EMPTY_LIST
import static java.util.Collections.EMPTY_MAP
import static org.ysb33r.grolifant5.api.core.StringTools.COMMA
import static org.ysb33r.grolifant5.api.core.StringTools.EMPTY

/**
 *
 * @author Schalk W. Cronjé
 *
 * @since 5.0
 */
@SuppressWarnings('AbstractClassWithoutAbstractMethod')
@CompileStatic
abstract class AbstractBaseGrolifantSimpleJavaForkOptions implements GrolifantSimpleJavaForkOptions, Serializable {

    final Provider<String> signature

    /**
     * Sets an alternative Java launcher.
     *
     * @param launcher Provider to a launcher.
     */
    @Override
    void setJavaLauncher(Provider<JavaLauncher> launcher) {
        this.javaExec.set(launcher.map { it.executablePath })
    }

    /**
     * Sets the system properties to use. Null values will be kept.
     *
     * <p>This will add to the system properties of the Java process.</p>
     *
     * @param sysProps System properties that will be lazy-evaluated.
     */
    @Override
    void setSystemProperties(Map<String, ?> sysProps) {
        systemProperties = ccso.stringTools().provideValuesKeepNull(sysProps)
    }

    /**
     * Sets the system properties to use.
     *
     * <p>This will add to the system properties of the Java process.</p>
     *
     * @param sysProps Provider to a map of system properties.
     */
    @Override
    void setSystemProperties(Provider<Map<String, String>> sysProps) {
        this.sysProps.set(sysProps.map {
            it.collectEntries {
                [it.key, it.value ?: EMPTY]
            } as Map<String, String>
        })
    }

    /**
     * Set the environment from the map. {@code null} values will be removed.
     *
     * <p>This will be added to the environment of the actual Java process.</p>
     *
     * @param env Environment that will be lazy-evaluated.
     */
    @Override
    void setEnvironment(Map<String, ?> env) {
        environment = ccso.stringTools().provideValuesDropNull(env)
    }

    /**
     * Set the environment from the provided map.
     *
     * <p>This will be added to the environment of the actual Java process.</p>
     *
     * @param env Environment that will be lazy-evaluated.
     */
    @Override
    void setEnvironment(Provider<Map<String, String>> env) {
        this.env.set(env.map {
            it.collectEntries {
                [it.key, it.value ?: EMPTY]
            } as Map<String, String>
        })
    }

    /**
     * Sets the JVM args to use. Null values are dropped.
     *
     * <p>This will add to the JVM arguments of the Java process.</p>
     *
     * @param args List of arguments. Will be lazy-evaluated.
     */
    @Override
    void setJvmArgs(Iterable<?> args) {
        jvmArgs = ccso.providerTools().provider { -> ccso.stringTools().stringizeDropNull(args.toList()) }
    }

    /**
     * Sets the JVM args to use.
     *
     * <p>This will add to the JVM arguments of the Java process.</p>
     *
     * @param args Provider to a list of arguments.
     */
    @Override
    void setJvmArgs(Provider<Iterable<String>> args) {
        this.jvmArguments.set(args.map { list -> list.findAll { it != null } as List<String> })
    }

    /**
     * Sets the maximum heap size.
     *
     * @param size Heap size.
     */
    @Override
    void setMaxHeapSize(String size) {
        this.maxHeap.set(ccso.providerTools().provider { -> size })
    }

    /**
     * Sets the maximum heap size.
     *
     * @param size Provider to a heap size.
     */
    @Override
    void setMaxHeapSize(Provider<String> size) {
        this.maxHeap.set(size)
    }

    /**
     * Sets the minimum heap size.
     *
     * @param size Heap size.
     */
    @Override
    void setMinHeapSize(String size) {
        this.minHeap.set(ccso.providerTools().provider { -> size })
    }

    /**
     * Sets the minimum heap size.
     *
     * @param size Provider to a heap size.
     */
    @Override
    void setMinHeapSize(Provider<String> size) {
        this.minHeap.set(size)
    }

    protected final RegularFileProperty javaExec
    protected final MapProperty<String, String> sysProps
    protected final MapProperty<String, String> env
    protected final ListProperty<String> jvmArguments
    protected final Property<String> maxHeap
    protected final Property<String> minHeap

    protected AbstractBaseGrolifantSimpleJavaForkOptions(Project tempProjectReference) {
        this.ccso = ConfigCacheSafeOperations.from(tempProjectReference)
        final objectFactory = tempProjectReference.objects
        this.javaExec = objectFactory.fileProperty()
        this.sysProps = objectFactory.mapProperty(String, String)
        this.env = objectFactory.mapProperty(String, String)
        this.jvmArguments = objectFactory.listProperty(String)
        this.maxHeap = objectFactory.property(String)
        this.minHeap = objectFactory.property(String)

        this.signature = sysProps.orElse(EMPTY_MAP).zip(env.orElse(EMPTY_MAP)) { x, y ->
            x.keySet().join(COMMA) + x.values().join(COMMA) + y.keySet().join(COMMA) + y.values().join(COMMA)
        }.zip(jvmArguments.orElse(EMPTY_LIST)) { x, y ->
            (x + y.join(COMMA)).md5()
        }
    }

    private final ConfigCacheSafeOperations ccso
}
