/**
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2025
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant5.api.core;

import org.gradle.api.Action;
import org.gradle.api.Project;
import org.gradle.api.file.ConfigurableFileTree;
import org.gradle.api.file.CopySpec;
import org.gradle.api.file.DeleteSpec;
import org.gradle.api.file.FileTree;
import org.gradle.api.logging.LogLevel;
import org.gradle.api.logging.configuration.ConsoleOutput;
import org.gradle.api.provider.Provider;
import org.gradle.api.provider.ProviderFactory;
import org.gradle.api.reflect.TypeOf;
import org.gradle.api.resources.ReadableResource;
import org.gradle.api.tasks.WorkResult;
import org.gradle.process.ExecResult;
import org.gradle.process.ExecSpec;
import org.gradle.process.JavaExecSpec;

import java.io.File;
import java.util.concurrent.Callable;

import static org.ysb33r.grolifant5.internal.core.DeprecatedProjectOperationsReporter.reportExt;

/**
 * A set of tools that can be used during the configuration phase.
 *
 * <p>
 * These tools MUST NOT be accessed from a task, except for the constructor itself, if needed.
 * No reference to it MUST BE stored in a task.
 * </p>
 *
 * @author Schalk W. Cronjé
 * @since 5.5
 */
public interface ConfigurationPhaseOperations extends ConfigCacheSafeOperations {

    /**
     * Create an instance from an existing {@link Project} reference.
     *
     * @param project Associated project.
     * @return Instance of instance.
     */
    static ConfigurationPhaseOperations from(Project project) {
        return project.getExtensions().getByType(TypeOf.typeOf(ConfigurationPhaseOperations.class));
    }

    /**
     * Access to a number of tools for manipulating Gradle configurations.
     *
     * @return Instance of a toolset.
     */
    ConfigurationTools configurationTools();

    /**
     * Access to a number of tools for working with task configuration.
     *
     * @return Instance of a toolset.
     */
    TaskTools taskTools();

    /**
     * Access to a number of tools for working with repositories
     *
     * @return Instance of a toolset.
     */
    RepositoryTools repositoryTools();

    /**
     * Lazy-evaluated project version.
     *
     * @return Provider to project version
     */
    default Provider<String> getVersionProvider() {
        return projectTools().getVersionProvider();
    }

    /**
     * Sets a new version for a project.
     * <p>This creates an internal objhect that can safely be evaluated by
     * Gradle's {@link org.gradle.api.Project#getVersion}.
     * </p>
     *
     * @param newVersion Anything that can be converted to a string using
     *                   {@link StringTools}.
     */
    default void setVersionProvider(Object newVersion) {
        projectTools().setVersionProvider(newVersion);
    }

    /**
     * Searches by Gradle property, then system property and finally by environment variable using the
     * {@code PropertyResolver convention}.
     *
     * @param name Anything convertible to a string
     * @return Provider to finding a property by the specified name.
     * The property cannot safely be read at configuration time.
     */
    default Provider<String> resolveProperty(Object name) {
        return providerTools().resolveProperty(name);
    }

    /**
     * Searches by Gradle property, then system property and finally by environment variable using the
     * {@code PropertyResolver convention}.
     *
     * @param name         Anything convertible to a string
     * @param defaultValue Default value to return if the property search order does not return any value.
     *                     Can be {@code null}. Anything convertible to a string.
     * @return Provider to finding a property by the specified name.
     * The property cannot safely be read at configuration time.
     */
    default Provider<String> resolveProperty(Object name, Object defaultValue) {
        return providerTools().resolveProperty(name, defaultValue);
    }

    /**
     * Creates a resource that points to a bzip2 compressed file at the given path.
     *
     * @param file File evaluated as per {@link #getFsOperations()}.{@link FileSystemOperations#file}.
     * @return Readable resource
     * @deprecated Use {@link #fsOperations()}.{@link FileSystemOperations#bzip2Resource(Object)}
     * Will be removed in 6.0
     */
    @Deprecated
    default ReadableResource bzip2Resource(Object file) {
        reportExt(
                this,
                "bzip2Resource will be removed in Grolifant 6.0",
                "Use grolifantOps.fsOperations().bzip2Resource"
        );
        return getFsOperations().bzip2Resource(file);
    }

    /**
     * Creates resource that points to a gzip compressed file at the given path.
     *
     * @param file File evaluated as per{@link #getFsOperations()}.{@link FileSystemOperations#file}.
     * @return Readable resource
     * @deprecated Use {@link #getFsOperations()}.{@link FileSystemOperations#gzipResource(Object)}
     * Will be removed in 6.0
     */
    @Deprecated
    default ReadableResource gzipResource(Object file) {
        reportExt(
                this,
                "gzipResource will be removed in Grolifant 6.0",
                "Use grolifantOps.fsOperations().gzip2Resource"
        );
        return getFsOperations().gzipResource(file);
    }

    /**
     * Creates an empty CopySpec.
     *
     * @return {@link CopySpec}
     * @deprecated Use {@link #getFsOperations}.{@link FileSystemOperations#copySpec()} instead.
     * Will be removed in 6.0
     */
    @Deprecated
    default CopySpec copySpec() {
        reportExt(
                this,
                "copySpec will be removed in Grolifant 6.0",
                "Use grolifantOps.fsOperations().copySpec"
        );
        return getFsOperations().copySpec();
    }

    /**
     * Creates a {@link CopySpec} which can later be used to copy files or create an archive. The given action is used
     * to configure the {@link CopySpec} before it is returned by this method.
     *
     * @param action Action to configure the CopySpec
     * @return The CopySpec
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default CopySpec copySpec(Action<? super CopySpec> action) {
        reportExt(
                this,
                "copySpec will be removed in Grolifant 6.0",
                "Use grolifantOps.fsOperations().copySpec"
        );
        return getFsOperations().copySpec(action);
    }

    /**
     * Safely resolve the stringy item as a path relative to the build directory.
     * <p>
     * Shortcut to {@link #getFsOperations()}.{@link FileSystemOperations#buildDirDescendant(Object)}.
     * Use the latter where possible.
     * </p>
     *
     * @param stringy Any item that can be resolved to a string using
     *                {@code org.ysb33r.grolifant5.core.api.grolifant.StringTools#stringize}.
     * @return Provider to a file
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default Provider<File> buildDirDescendant(Object stringy) {
        reportExt(
                this,
                "buildDirDescendant will be removed in Grolifant 6.0",
                "Use grolifantOps.fsOperations().buildDirDescendant"
        );
        return getFsOperations().buildDirDescendant(stringy);
    }

    /**
     * Deletes the specified files.  The given action is used to configure a {@link DeleteSpec}, which is then used to
     * delete the files.
     * <p>Example:
     * <pre>
     * project.delete {
     *     delete 'somefile'
     *     followSymlinks = true
     * }
     * </pre>
     *
     * @param action Action to configure the DeleteSpec
     * @return {@link WorkResult} that can be used to check if delete did any work.
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default WorkResult delete(Action<? super DeleteSpec> action) {
        reportExt(
                this,
                "delete will be removed in Grolifant 6.0",
                "Use grolifantOps.fsOperations().delete"
        );
        return getFsOperations().delete(action);
    }

    /**
     * Creates a provider to an environmental variable.
     *
     * @param name Anything convertible to a string
     * @return Provider to the value of the an environmental variable.
     * The property cannot safely be read at configuration time.
     * @since 1.1.0
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default Provider<String> environmentVariable(Object name) {
        reportExt(
                this,
                "delete will be removed in Grolifant 6.0",
                "Use grolifantOps.providerTools().environmentVariable or grolifantOps.resolveProperty"
        );
        return environmentVariable(name, false);
    }

    /**
     * Creates a provider to an environmental variable.
     *
     * @param name                    Anything convertible to a string
     * @param configurationTimeSafety Whether this property can be read safely at configuration time. As from Gradle 8.0
     *                                this value is ignored.
     * @return Provider to the value of an environment variable.
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default Provider<String> environmentVariable(Object name, boolean configurationTimeSafety) {
        reportExt(
                this,
                "delete will be removed in Grolifant 6.0",
                "Use grolifantOps.providerTools().environmentVariable or grolifantOps.resolveProperty"
        );
        return getProviderTools().environmentVariable(name, configurationTimeSafety);
    }

    /**
     * Executes the specified external process.
     *
     * @param action
     * @return {@link ExecResult} that can be used to check if the execution worked.
     * @deprecated Use {@link #getExecTools()}.{@link ExecTools#exec} instead.
     * Will be removed 6.0
     */
    @Deprecated
    default ExecResult exec(Action<? super ExecSpec> action) {
        reportExt(
                this,
                "exec will be removed in Grolifant 6.0",
                "Use grolifantOps.execTools().exec"
        );
        return getExecTools().exec(action).getResult().get();
    }

    /**
     * Provider for the build directory.
     *
     * @return Build directory.
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default Provider<File> getBuildDir() {
        reportExt(
                this,
                "getBuildDir will be removed in Grolifant 6.0",
                "Use project.layout.buildDirectory"
        );
        return fsOperations().buildDirDescendant(".");
    }

    /**
     * Tools to deal with Gradle configurations.
     *
     * @return
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default ConfigurationTools getConfigurations() {
        reportExt(
                this,
                "getConfigurations will be removed in Grolifant 6.0",
                "Use configurationTools()"
        );
        return configurationTools();
    }

    /**
     * Tools to deal with out-of-process, non-JVM, executables.
     *
     * @return Tools instance optimised for the current version of Gradle (where possible).
     * @deprecrated Will be removed in 6.0
     */
    @Deprecated
    default ExecTools getExecTools() {
        reportExt(
                this,
                "getExecTools will be removed in Grolifant 6.0",
                "Use execTools()"
        );
        return execTools();
    }

    /**
     * Get the full project path including the root project name in case of a multi-project.
     *
     * @return The fully qualified project path including the root project.
     * @since 1.2
     * @deprecated Use {@link ProjectTools#getFullProjectPath} instead.
     * Will be removed in 6.0
     */
    @Deprecated
    default String getFullProjectPath() {
        reportExt(
                this,
                "getFullProjectPath will be removed in Grolifant 6.0",
                "Use projectTools().getFullProjectPath()"
        );
        return projectTools().getFullProjectPath();
    }

    /**
     * Gradle distribution home directory.
     * <p>
     * Shortcut for {@link #getFsOperations()}.{@link FileSystemOperations#getGradleHomeDir}.
     * It is recommended to use the latter, unless this is access directly from a buildscript.
     * </p>
     *
     * @return Directory.
     * @since 2.2
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default Provider<File> getGradleHomeDir() {
        reportExt(
                this,
                "getGradleHomeDir will be removed in Grolifant 6.0",
                "Use fsOperations().getGradleHomeDir()"
        );
        return fsOperations().getGradleHomeDir();
    }

    /**
     * Gradle user home directory. Usually {@code ~/.gradle} on non-Windows.
     * <p>
     * Shortcut for {@link #getFsOperations()}.{@link FileSystemOperations#getGradleUserHomeDir}.
     * It is recommended to use the latter unless this is accessed directly from a buildscript.
     * </p>
     *
     * @return Directory.
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default Provider<File> getGradleUserHomeDir() {
        reportExt(
                this,
                "getGradleUserHomeDir will be removed in Grolifant 6.0",
                "Use fsOperations().getGradleUserHomeDir()"
        );
        return fsOperations().getGradleUserHomeDir();
    }

    /**
     * Lazy-evaluated project group.
     *
     * @return provider to project group
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default Provider<String> getGroupProvider() {
        reportExt(
                this,
                "getGroupProvider will be removed in Grolifant 6.0",
                "Use projectTools().getGroupProvider()"
        );
        return projectTools().getGroupProvider();
    }

    /**
     * Returns an object instance for filesystem operations that deals coprrectly with the functionality of the
     * curretn Gradle version.
     *
     * @return Instance of {@link FileSystemOperations}
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default FileSystemOperations getFsOperations() {
        reportExt(
                this,
                "getFsOperations will be removed in Grolifant 6.0",
                "Use fsOperations()"
        );
        return fsOperations();
    }

    /**
     * Returns the project cache directory for the given project.
     * <p>
     * Shortcut for {@link #getFsOperations()}.{@link FileSystemOperations#getProjectCacheDir}.
     * It is recommended to use the latter, unless this is access directly from a buildscript.
     * </p>
     *
     * @return Project cache directory. Never {@code null}.
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default File getProjectCacheDir() {
        reportExt(
                this,
                "getProjectCacheDir will be removed in Grolifant 6.0",
                "Use fsOperations().getProjectCacheDir()"
        );
        return fsOperations().getProjectCacheDir();
    }

    /**
     * Returns the project directory.
     *
     * <p>
     * Shortcut for {@link #getFsOperations()}.{@link FileSystemOperations#getProjectDir()}.
     * It is recommended to use the latter, unless this is access directly from a buildscript.
     * </p>
     *
     * @return Project directory.
     * @since 1.0
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default File getProjectDir() {
        reportExt(
                this,
                "getProjectDir will be removed in Grolifant 6.0",
                "Use fsOperations().getProjectDir()"
        );
        return fsOperations().getProjectDir();
    }

    /**
     * The project name
     *
     * @return Cached value of project name.
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default String getProjectName() {
        reportExt(
                this,
                "getProjectName will be removed in Grolifant 6.0",
                "Use projectTools().getProjectNameProvider()"
        );
        return projectTools().getProjectNameProvider().get();
    }

    /**
     * Get project path.
     *
     * @return The fully qualified project path.
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default String getProjectPath() {
        reportExt(
                this,
                "getProjectPath will be removed in Grolifant 6.0",
                "Use projectTools().getFullProjectPath()"
        );
        return projectTools().getFullProjectPath();
    }

    /**
     * Returns the root directory of the project.
     *
     * @return Root directory.
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default File getProjectRootDir() {
        reportExt(
                this,
                "getProjectRootDir will be removed in Grolifant 6.0",
                "Use fsOperations().getProjectRootDir()"
        );
        return fsOperations().getProjectRootDir();
    }

    /**
     * A reference to the provider factory.
     *
     * @return {@link ProviderFactory}
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    ProviderFactory getProviders();

    /**
     * Tools to deal with project & configuration specifics down to Gradle 4.0.
     *
     * @return Tools instance optimised for current version of Gradle (where possible).
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default ProjectTools getProjectTools() {
        reportExt(
                this,
                "getProjectTools will be removed in Grolifant 6.0",
                "Use projectTools()"
        );
        return projectTools();
    }

    /**
     * Tools to deal with provider down to Gradle 4.0.
     *
     * @return Tools instance optimised for current version of Gradle (where possible).
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default ProviderTools getProviderTools() {
        reportExt(
                this,
                "getProviderTools will be removed in Grolifant 6.0",
                "Use providerTools()"
        );
        return providerTools();
    }

    /**
     * Tools for dealing with repositories.
     *
     * @return Tools instance.
     * @since 2.0
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default RepositoryTools getRepositoryTools() {
        reportExt(
                this,
                "getRepositoryTools will be removed in Grolifant 6.0",
                "Use repositoryTools()"
        );
        return repositoryTools();
    }

    /**
     * Tools for dealing with conversions of various objects into string or lists of strings.
     *
     * @return String tools instance.
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default StringTools getStringTools() {
        reportExt(
                this,
                "getStringTools will be removed in Grolifant 6.0",
                "Use stringTools()"
        );
        return stringTools();
    }

    /**
     * Utilities for working with tasks in a consistent manner across Gradle versions.
     *
     * @return {@link TaskTools} instance.
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default TaskTools getTasks() {
        reportExt(
                this,
                "getTasks will be removed in Grolifant 6.0",
                "Use taskTools()"
        );
        return taskTools();
    }

    /**
     * Access to the problem reporter.
     *
     * @return Instance to something that can report problems.
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default GrolifantProblemReporter getProblemReporter() {
        reportExt(
                this,
                "getProblemReporter will be removed in Grolifant 6.0",
                "Use problemReporter()"
        );
        return problemReporter();
    }

    /**
     * Creates a new ConfigurableFileTree. The tree will have no base dir specified.
     *
     * @param base Base directory for file tree,
     * @return File tree.
     * @deprecated Use {@link #getFsOperations()}.{@link FileSystemOperations#fileTree(Object)} instead.
     * Will be removed in 6.0
     */
    @Deprecated
    default ConfigurableFileTree fileTree(Object base) {
        reportExt(
                this,
                "fileTree will be removed in Grolifant 6.0",
                "Use fsOperations().fileTree"
        );
        return fsOperations().fileTree(base);
    }

    /**
     * Console output mode
     *
     * @return How the console output has been requested.
     * @deprecated Use {@link ProjectTools#getConsoleOutput()}
     * Will be removed in 6.0
     */
    @Deprecated
    default ConsoleOutput getConsoleOutput() {
        reportExt(
                this,
                "getConsoleOutput will be removed in Grolifant 6.0",
                "Use projectTools().getConsoleOutput"
        );
        return projectTools().getConsoleOutput();
    }

    /**
     * Get the minimum log level for Gradle.
     *
     * @return Log level
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default LogLevel getGradleLogLevel() {
        reportExt(
                this,
                "getGradleLogLevel will be removed in Grolifant 6.0",
                "Use projectTools().getGradleLogLevel"
        );
        return projectTools().getGradleLogLevel();
    }

    /**
     * Creates a provider to a project property.
     *
     * @param name Anything convertible to a string
     * @return Provider to the value of the project property.
     * The property cannot safely be read at configuration time
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default Provider<String> gradleProperty(Object name) {
        reportExt(
                this,
                "gradleProperty will be removed in Grolifant 6.0",
                "Use providerTools().gradleProperty or resolveProperty"
        );
        return providerTools().gradleProperty(name);
    }

    /**
     * Creates a provider to a project property.
     *
     * @param name                    Anything convertible to a string
     * @param configurationTimeSafety Whether this property can be read safely at configuration time. As from Gradle 8.0
     *                                this value is ignored.
     * @return Provider to the value of the project property.
     * @deprecated Use {@link #getProviderTools}.{@link ProviderTools#gradleProperty} instead.
     * Will be removed in 6.0
     */
    @Deprecated
    default Provider<String> gradleProperty(Object name, boolean configurationTimeSafety) {
        reportExt(
                this,
                "gradleProperty will be removed in Grolifant 6.0",
                "Use providerTools().gradleProperty or resolveProperty"
        );
        return providerTools().gradleProperty(name, configurationTimeSafety);
    }

    /**
     * Whether configuration cache is enabled for a build.
     *
     * @return {@code true} is configuration cache is available and enabled.
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default boolean isConfigurationCacheEnabled() {
        reportExt(
                this,
                "isConfigurationCacheEnabled will be removed in Grolifant 6.0",
                "Use projectTools().isConfigurationCacheEnabled"
        );
        return projectTools().isConfigurationCacheEnabled();
    }

    /**
     * Whether Gradle is operating in offline mode.
     *
     * @return {@code true} if offline.
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default boolean isOffline() {
        reportExt(
                this,
                "isOffline will be removed in Grolifant 6.0",
                "Use projectTools().getOfflineProvider"
        );
        return projectTools().getOfflineProvider().get();
    }

    /**
     * Whether dependencies should be refreshed.
     *
     * @return {@code true} to check dependencies again.
     * @deprecated Use {@link ProjectTools#isRefreshDependencies()}.
     * Will be removed in 6.0
     */
    @Deprecated
    default boolean isRefreshDependencies() {
        reportExt(
                this,
                "isRefreshDependencies will be removed in Grolifant 6.0",
                "Use projectTools().isRefreshDependencies"
        );
        return projectTools().isRefreshDependencies();
    }


    /**
     * Whether tasks should be re-ruin
     *
     * @return {@code true} if tasks were set to be re-run.
     * @deprecated USe {@link ProjectTools#isRerunTasks()}
     * Will be removed in 6.0
     */
    @Deprecated
    default boolean isRerunTasks() {
        reportExt(
                this,
                "isRerunTasks will be removed in Grolifant 6.0",
                "Use projectTools().isRerunTasks()"
        );
        return projectTools().isRerunTasks();
    }

    /**
     * Whether current project is the root project.
     *
     * @return {@code true} is current project is root project.
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default boolean isRoot() {
        reportExt(
                this,
                "isRoot will be removed in Grolifant 6.0",
                "Use projectTools().isRootProject()"
        );
        return projectTools().isRootProject();
    }

    /**
     * Executes the specified external java process.
     *
     * @param action
     * @return {@link ExecResult} that can be used to check if the execution worked.
     * @deprecated Use {@link #getExecTools()}.{@link ExecTools#javaexec} instead.
     * Will be removed in 6.0
     */
    @Deprecated
    default ExecResult javaexec(Action<? super JavaExecSpec> action) {
        reportExt(
                this,
                "javaexec will be removed in Grolifant 6.0",
                "Use execTools().javaexec"
        );
        return execTools().javaexec(action).getResult().get();
    }

    /**
     * Tools for dealing for JVM/Java operations.
     *
     * @return Appropriate instance for the running Gradle version.
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default JvmTools getJvmTools() {
        reportExt(
                this,
                "getJvmTools will be removed in Grolifant 6.0",
                "Use jvmTools()"
        );
        return jvmTools();
    }

    /**
     * Searches by Gradle property, then system property and finally by environment variable using the
     * {@code PropertyResolver convention}.
     *
     * @param name                    Anything convertible to a string
     * @param configurationTimeSafety Whether this property can be read safely at configuration time. As from Gradle 8.0
     *                                this value is ignored.
     * @return Provider to finding a property by the specified name.
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default Provider<String> resolveProperty(Object name, boolean configurationTimeSafety) {
        reportExt(
                this,
                "resolveProperty(name,boolean) will be removed in Grolifant 6.0",
                "Use resolveProperty(name) instead."
        );
        return providerTools().resolveProperty(name, configurationTimeSafety);
    }

    /**
     * Searches by Gradle property, then system property and finally by environment variable using the
     * {@code PropertyResolver convention}.
     *
     * <p>
     * Shortcut to {@link #getProviderTools()}.{@link ProviderTools#resolveProperty(Object, Object, boolean)}.
     * </p>
     *
     * @param name                    Anything convertible to a string
     * @param defaultValue            Default value to return if the property search order does not return any value.
     *                                Can be {@code null}. Anything convertible to a string.
     * @param configurationTimeSafety Whether this property can be read safely at configuration time. As from Gradle 8.0
     *                                this value is ignored.
     * @return Provider to finding a property by the specified name.
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default Provider<String> resolveProperty(Object name, Object defaultValue, boolean configurationTimeSafety) {
        reportExt(
                this,
                "resolveProperty(name,boolean) will be removed in Grolifant 6.0",
                "Use resolveProperty(name) instead."
        );
        return providerTools().resolveProperty(name, defaultValue, configurationTimeSafety);
    }

    /**
     * Returns a provider.
     *
     * @param var1 Anything that adheres to a Callable including Groovy closures or Java lambdas.
     * @return Provider instance.
     * @deprecated Use {@link #getProviders()}.{@link ProviderTools#provider(Callable)} instead.
     * Will be removed in 6.0
     */
    @Deprecated
    default <T> Provider<T> provider(Callable<? extends T> var1) {
        reportExt(
                this,
                "provider will be removed in Grolifant 6.0",
                "Use providerTools().provider instead"
        );
        return providerTools().provider(var1);
    }

    /**
     * Creates a provider to a system property.
     * <p>
     * Shortcut to {@link #getProviderTools()}.{@link ProviderTools#systemProperty(Object)}.
     * </p>
     *
     * @param name Anything convertible to a string
     * @return Provider to the value of the system property. The property cannot safely be read at configuration time
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default Provider<String> systemProperty(Object name) {
        reportExt(
                this,
                "systemProperty will be removed in Grolifant 6.0",
                "Use project.providers.systemProperty or resolveProperty instead."
        );
        return systemProperty(name, false);
    }

    /**
     * Creates a provider to a system property.
     * <p>
     * Shortcut to {@link #getProviderTools()}.{@link ProviderTools#systemProperty(Object, boolean)}.
     * </p>
     *
     * @param name                    Anything convertible to a string
     * @param configurationTimeSafety Whether this property can be read safely at configuration time. As from Gradle 8.0
     *                                this value is ignored.
     * @return Provider to the value of the system property.
     * @deprecated Use {@link #getProviderTools()},{@link ProviderTools#systemProperty(Object)} instead.
     * Will be removed in 6.0
     */
    @Deprecated
    default Provider<String> systemProperty(Object name, boolean configurationTimeSafety) {
        reportExt(
                this,
                "systemProperty will be removed in Grolifant 6.0",
                "Use project.providers.systemProperty or resolveProperty instead"
        );
        return getProviderTools().systemProperty(name, configurationTimeSafety);
    }

    /**
     * Returns a TAR tree presentation
     * <p>
     * Shortcut for {@link #getFsOperations()}.{@link FileSystemOperations#tarTree(Object)}.
     * It is recommended to use the latter, unless this is access directly from a buildscript.
     * </p>
     *
     * @param tarPath Path to tar file
     * @return File tree
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default FileTree tarTree(Object tarPath) {
        reportExt(
                this,
                "tarTree will be removed in Grolifant 6.0",
                "Use fsOperations().tarTree"
        );
        return fsOperations().tarTree(tarPath);
    }

    /**
     * Returns a ZIP tree presentation
     * <p>
     * Shortcut for {@link #getFsOperations()}.{@link FileSystemOperations#zipTree(Object)}.
     * It is recommended to use the latter, unless this is access directly from a buildscript.
     * </p>
     *
     * @param zipPath Path to zip file
     * @return File tree
     * @deprecated Will be removed in 6.0
     */
    @Deprecated
    default FileTree zipTree(Object zipPath) {
        reportExt(
                this,
                "zipTree will be removed in Grolifant 6.0",
                "Use fsOperations().zipTree"
        );
        return fsOperations().zipTree(zipPath);
    }
}
