/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2025
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant5.internal.core.services

import groovy.transform.CompileStatic
import org.ysb33r.grolifant5.api.core.GrolifantProblemReporter
import org.ysb33r.grolifant5.api.errors.UnsupportedConfigurationException

import static org.ysb33r.grolifant5.internal.core.services.LegacyWarningReporterService.USE_LEGACY_WARNING_REPORTER

/**
 *
 * Handles problem reporting Pre-Gradle 8.6
 *
 * @author Schalk W. Cronjé
 *
 * @since 5.4
 */
@CompileStatic
class LegacyProblemDetails implements GrolifantProblemReporter.ProblemSpec {

    Optional<String> optContextualLabel = Optional.empty()
    Optional<String> optDetails = Optional.empty()
    Optional<String> optDocumentedAt = Optional.empty()
    Optional<String> optFileLocation = Optional.empty()
    Optional<String> optSeverity = Optional.empty()
    Optional<String> optSolution = Optional.empty()
    Optional<Throwable> optException = Optional.empty()

    LegacyProblemDetails() {
        if (!USE_LEGACY_WARNING_REPORTER) {
            throw new UnsupportedConfigurationException('This can only be used on Gradle versions < 8.11')
        }
    }

    @Override
    void setContextualLabel(String contextualLabel) {
        this.optContextualLabel = Optional.of(contextualLabel)
    }

    @Override
    void setDetails(String txt) {
        this.optDetails = Optional.of(txt)
    }

    @Override
    void setDocumentedAt(String url) {
        optDocumentedAt = Optional.of(url)
    }

    @Override
    void setFileLocation(String path) {
        optFileLocation = Optional.of(path)
    }

    @Override
    void lineInFileLocation(String path, int line) {
        optFileLocation = Optional.of("${path}:${line}".toString())
    }

    @Override
    void lineInFileLocation(String path, int line, int column) {
        optFileLocation = Optional.of("${path}:${line}:${column}".toString())
    }

    @Override
    void lineInFileLocation(String path, int line, int column, int length) {
        final end = column + length
        optFileLocation = Optional.of("${path}:${line}:${column}-${end}".toString())
    }

    @Override
    void offsetInFileLocation(String path, int offset, int length) {
        optFileLocation = Optional.of("${path}, offset=${offset}, len=${length}".toString())
    }

    @Override
    void setSeverity(GrolifantProblemReporter.Severity severity) {
        optSeverity = Optional.of(severity.name())
    }

    @Override
    void setSolution(String description) {
        optSolution = Optional.of(description)
    }

    @Override
    void stackLocation() {
    }

    @Override
    void withException(Throwable t) {
        optException = Optional.of(t)
    }
}
