/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2022
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.internal.v6.jvm

import groovy.transform.CompileStatic
import org.gradle.api.provider.Provider
import org.gradle.process.JavaExecSpec
import org.ysb33r.grolifant.api.core.ProjectOperations
import org.ysb33r.grolifant.api.core.jvm.ModularitySpec
import org.ysb33r.grolifant.api.errors.NotSupportedException

import static org.ysb33r.grolifant.api.core.LegacyLevel.PRE_6_4

/**
 * Proxy forward to modularity settings on Gradle 6.4+
 *
 * @authos Schalk W. Cronjé
 *
 * @since 2.0
 */
@CompileStatic
class DefaultModularitySpec implements ModularitySpec {
    DefaultModularitySpec(JavaExecSpec spec, ProjectOperations po) {
        if (PRE_6_4) {
            throw new NotSupportedException("${DefaultModularitySpec.canonicalName} needs Gradle 6.4+")
        }
        this.javaExecSpec = spec
        this.projectOperations = po
    }

    /**
     * Provider to the main module name.
     *
     * @return Provider.
     */
    @Override
    Provider<String> getMainModule() {
        javaExecSpec.mainModule
    }

    /**
     * Set the main module name.
     *
     * @param mod Anything convertible to a string.
     */
    @Override
    void setMainModule(Object mod) {
        projectOperations.stringTools.updateStringProperty(javaExecSpec.mainModule, mod)
    }

    /**
     * Whether module path should be inferred.
     *
     * Prior to Gradle 6.4, this will always return a false value.
     *
     * @return Provider to the detection state.
     */
    @Override
    Provider<Boolean> getInferModulePath() {
        javaExecSpec.modularity.inferModulePath
    }

    /**
     * Whether module path should be inferred.
     *
     * Ignored prior to Gradle 6.4.
     *
     * @param toggle Whether to infer the module path.
     */
    @Override
    void setInferModulePath(boolean toggle) {
        javaExecSpec.modularity.inferModulePath.set(toggle)
    }

    /**
     * Whether module path should be inferred.
     *
     * Ignored prior to Gradle 6.4.
     *
     * @param toggle Whether to infer the module path.
     */
    @Override
    void setInferModulePath(Provider<Boolean> toggle) {
        javaExecSpec.modularity.inferModulePath.set(toggle)
    }

    private final JavaExecSpec javaExecSpec
    private final ProjectOperations projectOperations
}
