/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2023
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.internal.v6.jvm

import groovy.transform.CompileStatic
import org.gradle.api.Action
import org.gradle.api.model.ObjectFactory
import org.gradle.api.tasks.Internal
import org.gradle.process.CommandLineArgumentProvider
import org.gradle.process.JavaExecSpec
import org.ysb33r.grolifant.api.core.CmdlineArgumentSpec
import org.ysb33r.grolifant.api.core.executable.ProcessExecutionSpec
import org.ysb33r.grolifant.api.core.ProjectOperations
import org.ysb33r.grolifant.api.core.jvm.JavaForkOptionsWithEnvProvider
import org.ysb33r.grolifant.api.core.jvm.JvmAppRunnerSpec
import org.ysb33r.grolifant.api.core.jvm.JvmEntryPoint
import org.ysb33r.grolifant.api.core.runnable.AbstractCmdlineArgumentSpec
import org.ysb33r.grolifant.api.errors.NotSupportedException
import org.ysb33r.grolifant.internal.core.runnable.EnvironmentVariableProviders
import org.ysb33r.grolifant.internal.core.runnable.ProcessExecutionSpecProxy
import org.ysb33r.grolifant.internal.v5.jvm.JavaForkOptionsWithEnvProviderProxy

import static org.ysb33r.grolifant.api.core.LegacyLevel.PRE_6_4
import static org.ysb33r.grolifant.internal.core.Transform.toList

/**
 * Internal class for common functionality related to {@code JavaExecSpec} configuration for Gradle 6.4+.
 *
 * Provider proxy functionality.
 *
 * @author Schalk W. Cronjé
 *
 * @since 2.0
 */
@CompileStatic
class InternalAbstractJvmAppExecSpec implements JvmAppRunnerSpec {

    /**
     * Creates the JvmExecSpec on Gradle 6.4+.
     *
     * @param po {@link ProjectOperations} instance
     * @param objects Object factory.
     * @param jfoProxyFactory Create a proxy instance for updating {@link org.gradle.process.JavaForkOptions}.
     * @param modularitySpecFactory Create a proxy instance for updating modularity specifications.
     */
    protected InternalAbstractJvmAppExecSpec(
        ProjectOperations po,
        ObjectFactory objects
    ) {
        if (PRE_6_4) {
            throw new NotSupportedException("${InternalAbstractJvmAppExecSpec.canonicalName} needs Gradle 6.4+")
        }

        this.javaExecSpec = po.jvmTools.javaExecSpec()
        this.arguments = new Arguments(po)
        this.projectOperations = po
        this.envProviders = new EnvironmentVariableProviders()
        this.jfoProxy = new JavaForkOptionsWithEnvProviderProxy(this.javaExecSpec, this.envProviders, objects)
        this.jepProxy = new JvmEntryPointProxy(this.javaExecSpec, po)
        this.psProxy = new ProcessExecutionSpecProxy(this.javaExecSpec)
    }

    @Override
    JavaExecSpec copyTo(JavaExecSpec options) {
        copyToJavaExecSpec(options)
        options
    }

    @Override
    void configureForkOptions(Action<JavaForkOptionsWithEnvProvider> configurator) {
        configurator.execute(this.jfoProxy)
    }

    @Override
    void configureCmdline(Action<CmdlineArgumentSpec> configurator) {
        configurator.execute(this.arguments)
    }

    @Override
    void configureEntrypoint(Action<JvmEntryPoint> configurator) {
        configurator.execute(this.jepProxy)
    }

    @Override
    void configureProcess(Action<ProcessExecutionSpec> configurator) {
        configurator.execute(this.psProxy)
    }

    /**
     * Holds a copy of the internal {@link JavaExecSpec}
     */
    @Internal
    protected final JavaExecSpec javaExecSpec

    @Internal
    protected final Arguments arguments

    @Internal
    protected final ProjectOperations projectOperations

    @Internal
    protected final EnvironmentVariableProviders envProviders

    @Internal
    protected final JavaForkOptionsWithEnvProvider jfoProxy

    @Internal
    protected final JvmEntryPointProxy jepProxy

    @Internal
    protected final ProcessExecutionSpecProxy psProxy

    /**
     * Copies all settings to a target {@link JavaExecSpec}.
     *
     * @param target Target {@link JavaExecSpec}
     */
    protected void copyToJavaExecSpec(JavaExecSpec target) {
        this.javaExecSpec.copyTo(target)
        toList(envProviders.environmentProviders) {
            it.get()
        }.each {
            target.environment(it)
        }
        copyArguments(target)
        copyDebugOptions(target)
        jepProxy.copyTo(target)
        target.ignoreExitValue = javaExecSpec.ignoreExitValue

        if (javaExecSpec.standardOutput != null) {
            target.standardOutput = javaExecSpec.standardOutput
        }

        if (javaExecSpec.standardInput != null) {
            target.standardInput = javaExecSpec.standardInput
        }

        if (javaExecSpec.errorOutput != null) {
            target.errorOutput = javaExecSpec.errorOutput
        }
    }

    /**
     * Copies command arguments (non-JVM) target {@link JavaExecSpec} and well as command providers.
     *
     * @param target Target {@link JavaExecSpec}.
     */
    protected void copyArguments(JavaExecSpec target) {
        target.args = arguments.args
        def cmdlineProviders = toList(arguments.commandLineArgumentProviders) { p ->
            new CommandLineArgumentProvider() {
                @Override
                Iterable<String> asArguments() {
                    p.get()
                }
            }
        }
        target.argumentProviders.addAll(cmdlineProviders)
    }

    /**
     * Copies debug options to target {@link JavaExecSpec}.
     *
     * @param targetTarget {@link JavaExecSpec}.
     */
    protected void copyDebugOptions(JavaExecSpec target) {
        def dopt = javaExecSpec.debugOptions
        target.debugOptions.identity {
            enabled.set(dopt.enabled)
            server.set(dopt.server)
            suspend.set(dopt.suspend)
            port.set(dopt.port)
        }
    }

    static protected class Arguments extends AbstractCmdlineArgumentSpec {
        Arguments(ProjectOperations po) {
            super(po.stringTools, po.providers)
        }
    }
}
