/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2023
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.internal.v6.jvm

import groovy.transform.CompileStatic
import org.gradle.api.model.ObjectFactory
import org.gradle.api.provider.Provider
import org.gradle.process.JavaExecSpec
import org.ysb33r.grolifant.api.core.ProjectOperations
import org.ysb33r.grolifant.api.core.jvm.ModularitySpec
import org.ysb33r.grolifant.internal.v5.jvm.InternalJvmAppRunnerSpec

/**
 * Handles JavaExec for Gradle 6.4 - Gradle 6.5.1.
 *
 * @since 2.0
 */
@CompileStatic
class InternalJavaAppRunnerSpecPre66 extends InternalJvmAppRunnerSpec {
    InternalJavaAppRunnerSpecPre66(
        ProjectOperations po,
        ObjectFactory objects
    ) {
        super(
            po,
            objects,
            { JavaExecSpec jes -> new ModularitySpecProxyPre66(jes, po) }
        )
    }

    /**
     * Copies all settings to a target {@link JavaExecSpec}.
     *
     * Since this is for Gradle 6.4 - 6.5.1, {@code mainClass, classpath, mainModule}, and {@code modularity} will
     * be updates as well.
     *
     * @param target Target {@link JavaExecSpec}
     */
    @Override
    protected void copyToJavaExecSpec(JavaExecSpec target) {
        super.copyToJavaExecSpec(target)
        target.mainClass.set(jepProxy.mainClass)
        target.mainModule.set(jepProxy.modularitySpecProxy.mainModule)
        target.modularity.inferModulePath.set(jepProxy.modularitySpecProxy.inferModulePath)
    }

    private static class ModularitySpecProxyPre66 implements ModularitySpec {

        ModularitySpecProxyPre66(JavaExecSpec jes, ProjectOperations po) {
            this.javaExecSpec = jes
            this.po = po
        }

        @Override
        Provider<String> getMainModule() {
            javaExecSpec.mainModule
        }

        @Override
        void setMainModule(Object mod) {
            po.stringTools.updateStringProperty(javaExecSpec.mainModule, mod)
        }

        @Override
        Provider<Boolean> getInferModulePath() {
            javaExecSpec.modularity.inferModulePath
        }

        @Override
        void setInferModulePath(boolean toggle) {
            javaExecSpec.modularity.inferModulePath.set(toggle)
        }

        @Override
        void setInferModulePath(Provider<Boolean> toggle) {
            javaExecSpec.modularity.inferModulePath.set(toggle)
        }

        private final JavaExecSpec javaExecSpec
        private final ProjectOperations po
    }
}
