/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2024
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.internal.v6.jvm

import groovy.transform.CompileStatic
import org.gradle.api.Action
import org.gradle.api.model.ObjectFactory
import org.gradle.api.tasks.Internal
import org.gradle.process.CommandLineArgumentProvider
import org.gradle.process.JavaExecSpec
import org.gradle.workers.WorkerExecutor
import org.ysb33r.grolifant.api.core.ClosureUtils
import org.ysb33r.grolifant.api.core.CmdlineArgumentSpec
import org.ysb33r.grolifant.api.core.ProjectOperations
import org.ysb33r.grolifant.api.core.executable.CmdLineArgumentSpecEntry
import org.ysb33r.grolifant.api.core.executable.ProcessExecutionSpec
import org.ysb33r.grolifant.api.core.jvm.JavaForkOptionsWithEnvProvider
import org.ysb33r.grolifant.api.core.jvm.JvmAppRunnerSpec
import org.ysb33r.grolifant.api.core.jvm.JvmEntryPoint
import org.ysb33r.grolifant.api.core.jvm.worker.WorkerAppParameterFactory
import org.ysb33r.grolifant.api.core.jvm.worker.WorkerExecSpec
import org.ysb33r.grolifant.api.core.jvm.worker.WorkerIsolation
import org.ysb33r.grolifant.api.core.jvm.worker.WorkerPromise
import org.ysb33r.grolifant.api.core.runnable.AbstractCmdlineArgumentSpec
import org.ysb33r.grolifant.api.errors.NotSupportedException
import org.ysb33r.grolifant.api.remote.worker.SerializableWorkerAppParameters
import org.ysb33r.grolifant.api.remote.worker.WorkerAppExecutorFactory
import org.ysb33r.grolifant.internal.core.runnable.EnvironmentVariableProviders
import org.ysb33r.grolifant.internal.core.runnable.ProcessExecutionSpecProxy
import org.ysb33r.grolifant.internal.v6.jvm.worker.WorkerSubmission

import java.security.MessageDigest

import static org.ysb33r.grolifant.api.core.LegacyLevel.PRE_6_4
import static org.ysb33r.grolifant.internal.core.Transform.toList

/**
 * Internal class for common functionality related to {@code JavaExecSpec} configuration for Gradle 6.4+.
 *
 * Provider proxy functionality.
 *
 * @author Schalk W. Cronjé
 *
 * @since 2.0
 */
@CompileStatic
class InternalAbstractJvmAppExecSpec implements JvmAppRunnerSpec {

    /**
     * Creates the JvmExecSpec on Gradle 6.4+.
     *
     * @param po {@link ProjectOperations} instance
     * @param objects Object factory.
     * @param jfoProxyFactory Create a proxy instance for updating {@link org.gradle.process.JavaForkOptions}.
     * @param modularitySpecFactory Create a proxy instance for updating modularity specifications.
     */
    protected InternalAbstractJvmAppExecSpec(
        ProjectOperations po,
        ObjectFactory objects
    ) {
        if (PRE_6_4) {
            throw new NotSupportedException("${InternalAbstractJvmAppExecSpec.canonicalName} needs Gradle 6.4+")
        }

        this.javaExecSpec = po.jvmTools.javaExecSpec()
        this.arguments = new Arguments(po)
        this.projectOperations = po
        this.envProviders = new EnvironmentVariableProviders()
        this.jfoProxy = new JavaForkOptionsWithEnvProviderProxy(this.javaExecSpec, this.envProviders, objects)
        this.jepProxy = new JvmEntryPointProxy(this.javaExecSpec, po)
        this.psProxy = new ProcessExecutionSpecProxy(this.javaExecSpec)
    }

    @Override
    JavaExecSpec copyTo(JavaExecSpec options) {
        copyToJavaExecSpec(options)
        options
    }

    @Override
    void configureForkOptions(Action<JavaForkOptionsWithEnvProvider> configurator) {
        configurator.execute(this.jfoProxy)
    }

    @Override
    void configureForkOptions(@DelegatesTo(JavaForkOptionsWithEnvProvider) Closure<?> configurator) {
        ClosureUtils.configureItem(this.jfoProxy, configurator)
    }

    @Override
    void configureCmdline(Action<CmdlineArgumentSpec> configurator) {
        configurator.execute(this.arguments)
    }

    @Override
    void configureCmdline(@DelegatesTo(CmdlineArgumentSpec) Closure<?> configurator) {
        ClosureUtils.configureItem(this.arguments, configurator)
    }

    @Override
    void configureEntrypoint(Action<JvmEntryPoint> configurator) {
        configurator.execute(this.jepProxy)
    }

    @Override
    void configureEntrypoint(@DelegatesTo(JvmEntryPoint) Closure<?> configurator) {
        ClosureUtils.configureItem(this.jepProxy, configurator)
    }

    @Override
    void configureProcess(Action<ProcessExecutionSpec> configurator) {
        configurator.execute(this.psProxy)
    }

    @Override
    void configureProcess(@DelegatesTo(ProcessExecutionSpec) Closure<?> configurator) {
        ClosureUtils.configureItem(this.psProxy, configurator)
    }

    /**
     * Submits this to a worker queue using an appropriate isolation mode.
     * @param isolationMode Isolation mode which is either classpath isolated or out of process.
     * @param worker A worker execution instance.
     * @param workerAppExecutorFactory A factory instances that can create executor logic.
     * @param parameterFactory A factory which can create parameters and populate them from a {@link JavaExecSpec}.
     * @param <P>          The type of the POJO/POGO that holds the parameters.
     */
    @Override
    public <P extends SerializableWorkerAppParameters> WorkerPromise submitToWorkQueue(
        WorkerIsolation isolationMode,
        WorkerExecutor worker,
        WorkerAppExecutorFactory<P> workerAppExecutorFactory,
        WorkerAppParameterFactory<P> parameterFactory
    ) {
        new WorkerSubmission<P>().toWorkQueue(
            isolationMode,
            worker,
            new InternalWorkerExecSpec(this),
            workerAppExecutorFactory,
            parameterFactory
        )
    }

    /**
     * Adds a command-line processor that will process command-line arguments in a specific order.
     *
     * For instance in a script, one want to proccess the exe args before the script args.
     *
     * In a system that has commands and subcommands, one wants to process this in the order of exe args, command args,
     * and then subcommand args.
     *
     * This method allows the implementor to control the order of processing  for all the groupings.
     *
     * @param name Name of command-line processor.
     * @param order Order in queue to process.
     * @param processor The specific grouping.
     */
    @Override
    void addCommandLineProcessor(String name, Integer order, CmdlineArgumentSpec processor) {
        this.cmdlineProcessors.add(new CmdLineArgumentSpecEntry(name, order, processor))
    }

    /**
     * Provides direct access to the list of command-line processors.
     *
     * In many cases there will only be one item in the list which is for providing arguments to the executable
     * itself. Some implementations will have more. Implementors can use this interface to manipulate order of
     * evaluation.
     *
     * @return Collection of command-line processors. Can be empty, but never {@code null}.
     */
    @Override
    Collection<CmdLineArgumentSpecEntry> getCommandLineProcessors() {
        this.cmdlineProcessors
    }

    /**
     * A unique string which determines whether there were any changes.
     *
     * @return Signature string
     */
    @Override
    String getExecutionSignature() {
        MessageDigest.getInstance('SHA-256')
            .digest(executionParameters.toString().bytes).sha256()
    }

    /**
     * Loads executions parameters from the current execution specification.
     * <p>
     *     The primary purpose of this method is to build a map for use by {@link #getExecutionSignature}.
     *     The default implementation will use the executable path and the arguments.
     *     </p>
     * @return Execution parameters.
     */
    @Internal
    protected Map<String, ?> getExecutionParameters() {
        final st = projectOperations.stringTools
        [
            exe             : st.stringize(javaExecSpec.executable),
            args1           : st.stringize(javaExecSpec.jvmArgs),
            args2           : st.stringize(javaExecSpec.jvmArgumentProviders*.asArguments().flatten()),
            args3           : st.stringize(javaExecSpec.args),
            args4           : st.stringize(javaExecSpec.argumentProviders*.asArguments().flatten()),
            systemProperties: st.stringizeValues(javaExecSpec.systemProperties),
            classpath       : javaExecSpec.classpath.asPath,
            main            : st.stringize(javaExecSpec.mainClass)
        ]
    }

    /**
     * Holds a copy of the internal {@link JavaExecSpec}
     */
    @Internal
    protected final JavaExecSpec javaExecSpec

    @Internal
    protected final Arguments arguments

    @Internal
    protected final ProjectOperations projectOperations

    @Internal
    protected final EnvironmentVariableProviders envProviders

    @Internal
    protected final JavaForkOptionsWithEnvProvider jfoProxy

    @Internal
    protected final JvmEntryPointProxy jepProxy

    @Internal
    protected final ProcessExecutionSpecProxy psProxy

    /**
     * Copies all settings to a target {@link JavaExecSpec}.
     *
     * Resolves environment then copies everything, because Gradle's API for the env does not recursively resolve
     * values in the env.
     *
     * @param target Target {@link JavaExecSpec}
     */
    protected void copyToJavaExecSpec(JavaExecSpec target) {
        this.javaExecSpec.copyTo(target)
        target.environment(projectOperations.stringTools.stringizeValues(this.javaExecSpec.environment))
        toList(envProviders.environmentProviders) {
            it.get()
        }.each {
            target.environment(it)
        }
        copyArguments(target)
        copyDebugOptions(target)
        jepProxy.copyTo(target)
        target.ignoreExitValue = javaExecSpec.ignoreExitValue

        if (javaExecSpec.standardOutput != null) {
            target.standardOutput = javaExecSpec.standardOutput
        }

        if (javaExecSpec.standardInput != null) {
            target.standardInput = javaExecSpec.standardInput
        }

        if (javaExecSpec.errorOutput != null) {
            target.errorOutput = javaExecSpec.errorOutput
        }
    }

    /**
     * Copies command arguments (non-JVM) target {@link JavaExecSpec} and well as command providers.
     *
     * @param target Target {@link JavaExecSpec}.
     */
    protected void copyArguments(JavaExecSpec target) {
        target.args = arguments.args
        def cmdlineProviders = toList(arguments.commandLineArgumentProviders) { p ->
            new CommandLineArgumentProvider() {
                @Override
                Iterable<String> asArguments() {
                    p.get()
                }
            }
        }
        target.argumentProviders.addAll(cmdlineProviders)
        commandLineProcessors.each {
            final providedArgs = it.argumentSpec.commandLineArgumentProviders

            if (it.order) {
                final allArgs = it.argumentSpec.allArgs
                target.argumentProviders.add({ -> allArgs.get() } as CommandLineArgumentProvider)
            } else {
                target.args(it.argumentSpec.args)

                providedArgs.each { p ->
                    target.argumentProviders.add({ -> p.get() } as CommandLineArgumentProvider)
                }
            }
        }
    }

    /**
     * Builds a list of arguments by taking all the set arguments as well as the argument providers.
     *
     * THe main purpose of this method is to provide a list of arguments to be passed to a Worker instance.
     *
     * @return All application arguments
     */
    protected List<String> buildArguments() {
        final List<String> result = []
        final tmpSpec = projectOperations.jvmTools.javaExecSpec()
        copyArguments(tmpSpec)
        result.addAll(tmpSpec.args)
        result.addAll(tmpSpec.argumentProviders*.asArguments().flatten().toList() as List<String>)
        result
    }

    /**
     * Copies debug options to target {@link JavaExecSpec}.
     *
     * @param targetTarget {@link JavaExecSpec}.
     */
    protected void copyDebugOptions(JavaExecSpec target) {
        def dopt = javaExecSpec.debugOptions
        target.debugOptions.identity {
            enabled.set(dopt.enabled)
            server.set(dopt.server)
            suspend.set(dopt.suspend)
            port.set(dopt.port)
        }
    }

    static protected class Arguments extends AbstractCmdlineArgumentSpec {
        Arguments(ProjectOperations po) {
            super(po.stringTools, po.providers)
        }
    }

    static private class InternalWorkerExecSpec implements WorkerExecSpec {
        InternalWorkerExecSpec(InternalAbstractJvmAppExecSpec thisRef) {
            this.ref = thisRef
        }

        @Override
        JavaExecSpec getJavaExecSpec() {
            ref.javaExecSpec
        }

        @Override
        JvmEntryPoint getJvmEntrypoint() {
            ref.jepProxy
        }

        @Override
        List<String> getApplicationArguments() {
            ref.buildArguments()
        }

        private final InternalAbstractJvmAppExecSpec ref
    }

    private final SortedSet<CmdLineArgumentSpecEntry> cmdlineProcessors = [] as SortedSet<CmdLineArgumentSpecEntry>
}
