/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2024
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.internal.v6.jvm.pre64

import groovy.transform.CompileStatic
import groovy.util.logging.Slf4j
import org.gradle.api.Project
import org.gradle.api.provider.Provider
import org.gradle.util.GradleVersion
import org.ysb33r.grolifant.api.core.ProjectOperations
import org.ysb33r.grolifant.api.core.jvm.ModularitySpec

import java.util.concurrent.Callable

/**
 * Allows modularity specs to be used on Gradle < 6.4, but ignore the settings.
 *
 * @author Schalk W> Cronjé
 *
 * @since 4.0
 */
@CompileStatic
@Slf4j
class FakeModularitySpec implements ModularitySpec {

    FakeModularitySpec(Project po) {
        infer = po.provider { -> false }
        module = po.provider({ -> null } as Callable<String>)
        gradleVersion = GradleVersion.current().version
    }

    FakeModularitySpec(ProjectOperations po) {
        infer = po.provider { -> false }
        module = po.provider({ -> null } as Callable<String>)
        gradleVersion = GradleVersion.current().version
    }

    /**
     * Provider to the main module name.
     *
     * @return Provider.
     */
    @Override
    Provider<String> getMainModule() {
        module
    }

    /**
     * Set the main module name.
     *
     * @param mod Anything convertible to a string.
     */
    @Override
    void setMainModule(Object mod) {
        log.warn("Module path cannot be set on Gradle ${gradleVersion}. setMainModule is ignored.")
    }

    /**
     * Whether module path should be inferred.
     *
     * Prior to Gradle 6.4, this will always return a false value.
     *
     * @return Provider to the detection state.
     */
    @Override
    Provider<Boolean> getInferModulePath() {
        infer
    }

    /**
     * Whether module path should be inferred.
     *
     * Ignored prior to Gradle 6.4.
     *
     * @param toggle Whether to infer the module path.
     */
    @Override
    void setInferModulePath(boolean toggle) {
        log.warn("Module path cannot be inferred on Gradle ${gradleVersion}. setInferModulePath is ignored.")
    }

    /**
     * Whether module path should be inferred.
     *
     * Ignored prior to Gradle 6.4.
     *
     * @param toggle Whether to infer the module path.
     */
    @Override
    void setInferModulePath(Provider<Boolean> toggle) {
        log.warn("Module path cannot be inferred on Gradle ${gradleVersion}. setInferModulePath is ignored.")
    }

    private final String gradleVersion
    private final Provider<Boolean> infer
    private final Provider<String> module
}
