/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2024
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.internal.v6.jvm.pre64

import groovy.transform.CompileStatic
import groovy.util.logging.Slf4j
import org.gradle.api.file.ConfigurableFileCollection
import org.gradle.api.file.FileCollection
import org.gradle.api.tasks.Internal
import org.gradle.process.BaseExecSpec
import org.gradle.process.JavaExecSpec
import org.gradle.process.JavaForkOptions
import org.gradle.process.ProcessForkOptions
import org.gradle.util.GradleVersion
import org.ysb33r.grolifant.api.core.OperatingSystem
import org.ysb33r.grolifant.api.core.ProjectOperations
import org.ysb33r.grolifant.api.core.runnable.AbstractCmdlineArgumentSpec
import org.ysb33r.grolifant.api.errors.NotSupportedException
import org.ysb33r.grolifant.internal.core.IterableUtils

import javax.annotation.Nullable

@CompileStatic
@Slf4j
@SuppressWarnings('MethodCount')
class Pre46FakeJavaExecSpec {

    Pre46FakeJavaExecSpec(ProjectOperations po) {
        this.projectOperations = po
        this.arguments = new Arguments(po)
        this.jvmArguments = new Arguments(po)
        this.bootstrap = po.fsOperations.emptyFileCollection()
        this.cp = po.fsOperations.emptyFileCollection()
    }

    String getMain() {
        this.mainClass
    }

    JavaExecSpec setMain(@Nullable String s) {
        this.mainClass = s
        me()
    }

    List<String> getArgs() {
        this.arguments.args
    }

    JavaExecSpec args(Object... objects) {
        this.arguments.args(objects)
        me()
    }

    JavaExecSpec args(Iterable<?> iterable) {
        this.arguments.args(iterable)
        me()
    }

    @SuppressWarnings('UnnecessarySetter')
    JavaExecSpec setArgs(@Nullable List<String> list) {
        this.arguments.setArgs(list)
        me()
    }

    @SuppressWarnings('UnnecessarySetter')
    JavaExecSpec setArgs(@Nullable Iterable<?> iterable) {
        this.arguments.setArgs(iterable)
        me()
    }

    JavaExecSpec classpath(Object... objects) {
        this.cp.from(objects)
        me()
    }

    FileCollection getClasspath() {
        this.cp
    }

    JavaExecSpec setClasspath(FileCollection fileCollection) {
        this.cp = projectOperations.fsOperations.emptyFileCollection()
        this.cp.from(fileCollection)
        me()
    }

    boolean getDebug() {
        this.debug
    }

    void setDebug(boolean dbg) {
        this.debug = dbg
    }

    BaseExecSpec setIgnoreExitValue(boolean b) {
        this.ignoreExit = b
        me()
    }

    boolean isIgnoreExitValue() {
        this.ignoreExit
    }

    BaseExecSpec setStandardInput(InputStream inputStream) {
        this.inputStream = inputStream
        me()
    }

    InputStream getStandardInput() {
        this.inputStream
    }

    BaseExecSpec setStandardOutput(OutputStream outputStream) {
        this.outputStream = outputStream
        me()
    }

    OutputStream getStandardOutput() {
        this.outputStream
    }

    BaseExecSpec setErrorOutput(OutputStream outputStream) {
        this.errorStream = outputStream
        me()
    }

    OutputStream getErrorOutput() {
        this.errorStream
    }

    List<String> getCommandLine() {
        throw new NotSupportedException("getCommandLine is not supported on Gradle ${GradleVersion.current().version}")
    }

    Map<String, Object> getSystemProperties() {
        this.sysProps
    }

    void setSystemProperties(Map<String, ?> map) {
        this.sysProps.clear()
        this.sysProps.putAll(map)
    }

    JavaForkOptions systemProperties(Map<String, ?> map) {
        this.sysProps.putAll(map)
        me()
    }

    JavaForkOptions systemProperty(String s, Object o) {
        this.sysProps.put(s, o)
        me()
    }

    String getDefaultCharacterEncoding() {
        this.characterEncoding
    }

    void setDefaultCharacterEncoding(@Nullable String s) {
        this.characterEncoding = s
    }

    String getMinHeapSize() {
        this.minHeapSize
    }

    void setMinHeapSize(@Nullable String s) {
        this.minHeapSize = s
    }

    String getMaxHeapSize() {
        this.maxHeapSize
    }

    void setMaxHeapSize(@Nullable String s) {
        this.maxHeapSize = s
    }

    List<String> getJvmArgs() {
        jvmArguments.args
    }

    void setJvmArgs(@Nullable List<String> list) {
        jvmArguments.args = list
    }

    void setJvmArgs(@Nullable Iterable<?> iterable) {
        jvmArguments.args = iterable
    }

    JavaForkOptions jvmArgs(Iterable<?> iterable) {
        jvmArguments.args(iterable)
        me()
    }

    JavaForkOptions jvmArgs(Object... objects) {
        jvmArguments.args(objects)
        me()
    }

    FileCollection getBootstrapClasspath() {
        this.bootstrap
    }

    void setBootstrapClasspath(FileCollection fileCollection) {
        this.bootstrap = projectOperations.fsOperations.emptyFileCollection()
        this.bootstrap.from(fileCollection)
    }

    JavaForkOptions bootstrapClasspath(Object... objects) {
        this.bootstrap.from(objects)
        me()
    }

    boolean getEnableAssertions() {
        this.assertions
    }

    void setEnableAssertions(boolean b) {
        this.assertions = b
    }

    List<String> getAllJvmArgs() {
        jvmArguments.allArgs.get()
    }

    void setAllJvmArgs(List<String> list) {
        replaceSysPropsFrom(list)
        replaceBootstrapClasspathFrom(list)

        def minHeap = list.find { it.startsWith(MINHEAP_PARAM) }
        if (minHeap) {
            minHeapSize = projectOperations.stringTools.stringize(minHeap)
                .replaceFirst(/${MINHEAP_PARAM}/, '')
        }
        def maxHeap = list.find { it.startsWith(MAXHEAP_PARAM) }
        if (maxHeap) {
            maxHeapSize = projectOperations.stringTools.stringize(maxHeap)
                .replaceFirst(/${MAXHEAP_PARAM}/, '')
        }

        jvmArgs = list.findAll {
            !it.startsWith(SYSPROP_PARAM) &&
                !it.startsWith(BOOTCLASSPATH_PARAM) &&
                !it.startsWith(MINHEAP_PARAM) &&
                !it.startsWith(MAXHEAP_PARAM)
        }
    }

    void setAllJvmArgs(Iterable<?> iterable) {
        allJvmArgs = projectOperations.stringTools.stringizeDropNull(IterableUtils.toList((Iterable) iterable))
    }

    ProcessForkOptions copyTo(ProcessForkOptions processForkOptions) {
        processForkOptions.environment = environment

        if (executable != null) {
            processForkOptions.executable = executable
        }
        if (workingDir != null) {
            processForkOptions.workingDir = workingDir
        }
        me()
    }

    JavaForkOptions copyTo(JavaForkOptions javaForkOptions) {
        copyTo((ProcessForkOptions) javaForkOptions)
        javaForkOptions.bootstrapClasspath = bootstrapClasspath
        javaForkOptions.systemProperties = systemProperties

        if (debug != null) {
            javaForkOptions.debug = debug
        }

        if (this.characterEncoding) {
            javaForkOptions.defaultCharacterEncoding = defaultCharacterEncoding
        }
        if (assertions != null) {
            javaForkOptions.enableAssertions = enableAssertions
        }

        if (this.minHeapSize) {
            javaForkOptions.minHeapSize = minHeapSize
        }

        if (this.maxHeapSize) {
            javaForkOptions.maxHeapSize = maxHeapSize
        }

        //        javaForkOptions.allJvmArgs = allJvmArgs
        javaForkOptions.jvmArgs = jvmArgs // what about argument providers?

        me()
    }

    String getExecutable() {
        projectOperations.stringTools.stringizeOrNull(this.exe)
    }

    void setExecutable(String s) {
        this.exe = s
    }

    void setExecutable(Object o) {
        this.exe = o
    }

    ProcessForkOptions executable(Object o) {
        this.exe = o
        me()
    }

    File getWorkingDir() {
        projectOperations.fsOperations.fileOrNull(this.wd)
    }

    void setWorkingDir(File file) {
        this.wd = file
    }

    void setWorkingDir(Object o) {
        this.wd = o
    }

    ProcessForkOptions workingDir(Object o) {
        this.wd = o
        me()
    }

    Map<String, Object> getEnvironment() {
        this.env
    }

    void setEnvironment(Map<String, ?> map) {
        env.clear()
        env.putAll(map)
    }

    ProcessForkOptions environment(Map<String, ?> map) {
        env.putAll(map)
        me()
    }

    ProcessForkOptions environment(String s, Object o) {
        env.put(s, o)
        me()
    }

    protected JavaExecSpec me() {
        this as JavaExecSpec
    }

    @Internal
    protected final Arguments arguments

    @Internal
    protected final Arguments jvmArguments

    @Internal
    protected final ProjectOperations projectOperations

    @Internal
    protected String mainClass

    private void replaceSysPropsFrom(List<String> list) {
        systemProperties = list.findAll { it.startsWith(SYSPROP_PARAM) }
            *.replaceFirst(/^${SYSPROP_PARAM}/, '')
            .collectEntries {
                def parts = it.split(/=/, 2)
                [parts[0], parts.size() == 1 ? '' : parts[1]]
            }
    }

    private void replaceBootstrapClasspathFrom(final List<String> list) {
        def bc = list.find { it.startsWith('-Xbootclasspath:') }
        if (bc) {
            bootstrapClasspath = projectOperations.fsOperations.files(
                bc.replaceFirst(/^${BOOTCLASSPATH_PARAM}/, '')
                    .split(/${OperatingSystem.current().pathSeparator}/)
                    .toList()
            )
        }
    }

    private ConfigurableFileCollection bootstrap
    private ConfigurableFileCollection cp

    private boolean assertions
    private String characterEncoding
    private boolean debug = false
    private boolean ignoreExit = false
    private String minHeapSize
    private String maxHeapSize
    private Object exe
    private Object wd
    private OutputStream outputStream
    private OutputStream errorStream
    private InputStream inputStream
    private final Map<String, ?> env = [:]
    private final Map<String, Object> sysProps = [:]

    private static final String BOOTCLASSPATH_PARAM = '-Xbootclasspath:'
    private static final String SYSPROP_PARAM = '-D'
    private static final String MINHEAP_PARAM = '-Xms'
    private static final String MAXHEAP_PARAM = '-Xmx'

    static private class Arguments extends AbstractCmdlineArgumentSpec {
        Arguments(ProjectOperations po) {
            super(po.stringTools, po.providers)
        }
    }
}
