/*
 * ============================================================================
 * (C) Copyright Schalk W. Cronje 2016 - 2023
 *
 * This software is licensed under the Apache License 2.0
 * See http://www.apache.org/licenses/LICENSE-2.0 for license details
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 * ============================================================================
 */
package org.ysb33r.grolifant.internal.v7.jvm

import groovy.transform.CompileStatic
import org.gradle.api.model.ObjectFactory
import org.gradle.process.CommandLineArgumentProvider
import org.gradle.process.JavaExecSpec
import org.ysb33r.grolifant.api.core.ProjectOperations
import org.ysb33r.grolifant.api.core.jvm.JavaForkOptionsWithEnvProvider
import org.ysb33r.grolifant.internal.core.runnable.EnvironmentVariableProviders
import org.ysb33r.grolifant.internal.v4.jvm.InternalAbstractJvmAppRunnerSpec
import org.ysb33r.grolifant.internal.v5.jvm.JavaForkOptionsWithEnvProviderProxy

import static org.ysb33r.grolifant.internal.core.Transform.toList

/**
 * Provides a class that can be populated with various fork options for Java
 * and which can then be used to copy to other methods in the Gradle API that provides a
 * {@link org.gradle.process.JavaForkOptions} in the parameters.
 *
 * Intended for Gradle 7.x
 *
 * @author Schalk W. Cronjé
 *
 * @since 2.0
 */
@CompileStatic
class JvmAppExecSpec extends InternalAbstractJvmAppRunnerSpec {

    JvmAppExecSpec(ProjectOperations po, ObjectFactory objects) {
        super(
            po,
            { JavaExecSpec jes, EnvironmentVariableProviders evp -> createJfoProxy(jes, evp, objects) },
            { JavaExecSpec jes -> new DefaultModularitySpec(jes, po) }
        )
    }

    /**
     * Copies command arguments (non-JVM) target {@link JavaExecSpec}.
     *
     * @param target Target {@link JavaExecSpec}.
     */
    @Override
    void copyArguments(JavaExecSpec target) {
        target.args = arguments.args
        def cmdlineProviders = toList(arguments.commandLineArgumentProviders) { p ->
            new CommandLineArgumentProvider() {
                @Override
                Iterable<String> asArguments() {
                    p.get()
                }
            }
        }
        target.argumentProviders.addAll(cmdlineProviders)
    }

    /**
     * Copies debug options to target {@link JavaExecSpec}.
     *
     * @param targetTarget {@link JavaExecSpec}.
     */
    @Override
    void copyDebugOptions(JavaExecSpec target) {
        def dopt = javaExecSpec.debugOptions
        target.debugOptions.identity {
            enabled.set(dopt.enabled)
            server.set(dopt.server)
            suspend.set(dopt.suspend)
            port.set(dopt.port)
        }
    }

    private static JavaForkOptionsWithEnvProvider createJfoProxy(
        JavaExecSpec es,
        EnvironmentVariableProviders evp,
        ObjectFactory objects
    ) {
        new JavaForkOptionsWithEnvProviderProxy(es, evp, objects)
    }

}
