package org.zalando.emsig;

import org.zalando.emsig.exception.InvalidSignatureException;
import org.zalando.emsig.exception.MissingSignatureException;
import org.zalando.emsig.exception.SignatureException;
import org.zalando.emsig.model.Signable;
import org.zalando.emsig.model.Signature;

import java.util.ArrayList;
import java.util.List;

public final class DefaultSignatureValidator implements SignatureValidator {
    private final List<Signer> signers;

    public DefaultSignatureValidator(final List<DefaultSigner> signers) {
        this.signers = new ArrayList<Signer>(signers);
    }

    @Override
    public void validate(final Signable signable) throws SignatureException {
        final Signature promisedSignature = signable.getSignature();

        if (promisedSignature == null) {
            throw new MissingSignatureException();
        }

        final String promisedSignatureValue = promisedSignature.getValue();
        if (promisedSignatureValue == null || promisedSignatureValue.length() == 0) {
            throw new MissingSignatureException();
        }

        boolean valid = false;
        for (Signer signer : signers) {
            final Signature actualSignature = signer.sign(signable);
            final String actualSignatureValue = actualSignature.getValue();

            valid |= constantTimeEquals(promisedSignatureValue, actualSignatureValue);
        }

        if (!valid) {
            throw new InvalidSignatureException();
        }
    }

    private static boolean constantTimeEquals(String promisedSignatureValue, String actualSignatureValue) {
        final int actualLength = actualSignatureValue.length();
        final int promisedLength = promisedSignatureValue.length();

        boolean equals = promisedLength == actualLength;

        for (int i = 0; i < Math.min(promisedLength, actualLength); i++) {
            equals &= promisedSignatureValue.charAt(i) == actualSignatureValue.charAt(i);
        }

        return equals;
    }

}
