package org.zalando.emsig;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import org.zalando.emsig.model.Signable;
import org.zalando.emsig.model.Signature;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;

public final class DefaultSigner implements Signer {
    private static final char[] HEX_CHARS = "0123456789abcdef".toCharArray();

    private final ObjectWriter canonicalObjectWriter;
    private final SecretKeySpec key;

    public DefaultSigner(final ObjectMapper objectMapper, final SignatureConfiguration configuration) {
        this(Emsig.canonicalObjectWriter(objectMapper), configuration);
    }

    public DefaultSigner(final ObjectWriter canonicalObjectWriter, final SignatureConfiguration configuration) {
        this.canonicalObjectWriter = canonicalObjectWriter;
        this.key = new SecretKeySpec(configuration.getPassphraseBytes(), configuration.getInternalAlgorithmName());
    }

    @Override
    public Signature sign(final Signable signable) {
        try {
            final byte[] canonicalBytes = serializeCanonicalJson(signable);

            final String algorithm = key.getAlgorithm();
            final Mac mac = Mac.getInstance(algorithm);
            mac.init(key);
            final byte[] signatureBytes = mac.doFinal(canonicalBytes);

            return new Signature(toHex(signatureBytes));
        } catch (NoSuchAlgorithmException | InvalidKeyException e) {
            throw new IllegalStateException(e);
        }
    }

    byte[] serializeCanonicalJson(final Signable signable) {
        try {
            return canonicalObjectWriter.writeValueAsBytes(signable);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException(e);
        }
    }

    private static String toHex(final byte[] data) {
        final StringBuilder r = new StringBuilder(data.length * 2);
        for (byte b : data) {
            r.append(HEX_CHARS[(b >> 4) & 0xF]);
            r.append(HEX_CHARS[(b & 0xF)]);
        }
        return r.toString();
    }
}
