package org.zalando.emsig;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.databind.SerializationFeature;
import org.zalando.emsig.exception.SignatureException;
import org.zalando.emsig.model.IncludeInSignature;
import org.zalando.emsig.model.Signable;
import org.zalando.emsig.model.Signature;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

public final class Emsig implements Signer, SignatureValidator {

    private final DefaultSigner signer;
    private final DefaultSignatureValidator validator;

    public Emsig(final ObjectMapper objectMapper, final String passphrase) {
        this(objectMapper, new SignatureConfiguration(SignatureConfiguration.DEFAULT_ALGORITHM, passphrase));
    }

    public Emsig(final ObjectMapper objectMapper, final SignatureConfiguration configuration) {
        this(objectMapper, configuration, Collections.singletonList(configuration));
    }

    public Emsig(final ObjectMapper objectMapper, final SignatureConfiguration signingConfiguration, final List<SignatureConfiguration> validationConfigurations) {
        final ObjectWriter canonicalObjectWriter = canonicalObjectWriter(objectMapper);
        final List<DefaultSigner> signers = createSigners(canonicalObjectWriter, validationConfigurations);

        this.signer = new DefaultSigner(canonicalObjectWriter, signingConfiguration);
        this.validator = new DefaultSignatureValidator(signers);
    }

    private static List<DefaultSigner> createSigners(final ObjectWriter canonicalObjectWriter, final List<SignatureConfiguration> configurations) {
        final List<DefaultSigner> delegates = new ArrayList<>(configurations.size());
        for (SignatureConfiguration configuration : configurations) {
            delegates.add(new DefaultSigner(canonicalObjectWriter, configuration));
        }
        return delegates;
    }

    static ObjectWriter canonicalObjectWriter(final ObjectMapper objectMapper) {
        final ObjectMapper copiedObjectMapper = objectMapper.copy();
        copiedObjectMapper.disable(MapperFeature.DEFAULT_VIEW_INCLUSION);

        copiedObjectMapper.enable(MapperFeature.SORT_PROPERTIES_ALPHABETICALLY);
        copiedObjectMapper.disable(SerializationFeature.INDENT_OUTPUT);

        copiedObjectMapper.enable(DeserializationFeature.USE_BIG_DECIMAL_FOR_FLOATS);

        //Making signature independent to date-time representation format
        copiedObjectMapper.enable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);
        copiedObjectMapper.setDefaultPropertyInclusion(JsonInclude.Value.construct(JsonInclude.Include.ALWAYS, JsonInclude.Include.NON_NULL));

        copiedObjectMapper.setDefaultPrettyPrinter(null);

        return copiedObjectMapper.writerWithView(IncludeInSignature.class)
                .with(JsonGenerator.Feature.WRITE_BIGDECIMAL_AS_PLAIN)
                .with(JsonGenerator.Feature.ESCAPE_NON_ASCII);
    }

    @Override
    public Signature sign(final Signable signable) {
        return signer.sign(signable);
    }

    @Override
    public void validate(final Signable signable) throws SignatureException {
        validator.validate(signable);
    }
}
