package org.zalando.emsig;

import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.Map;

public final class SignatureConfiguration {
    /**
     * Supported signature internalAlgorithmName names.
     *
     * @see <a href="https://tools.ietf.org/html/draft-ietf-jose-json-web-algorithms-40#section-3.1">JSON Web Algorithms (JWA)</a>
     */
    public static final String HMAC_SHA256 = "HS256";
    public static final String DEFAULT_ALGORITHM = HMAC_SHA256;

    private static final String HMAC_SHA256_INTERNAL_NAME = "HmacSHA256";
    private static final Map<String, String> TO_INTERNAL_NAME = Collections.singletonMap(HMAC_SHA256, HMAC_SHA256_INTERNAL_NAME);
    private static final Map<String, String> TO_EXTERNAL_NAME = Collections.singletonMap(HMAC_SHA256_INTERNAL_NAME, HMAC_SHA256);

    private final String algorithm;
    private final String internalAlgorithmName;
    private final String passphrase;

    public SignatureConfiguration(final String passphrase) {
        this(DEFAULT_ALGORITHM, passphrase);
    }

    public SignatureConfiguration(final String algorithm, final String passphrase) {
        final String internalAlgorithmName = TO_INTERNAL_NAME.get(algorithm);
        if (internalAlgorithmName == null) {
            throw new IllegalArgumentException("Unsupported algorithm [" + algorithm + "]");
        }

        this.algorithm = algorithm;
        this.internalAlgorithmName = internalAlgorithmName;
        this.passphrase = passphrase;
    }

    public String getAlgorithm() {
        return algorithm;
    }

    public String getPassphrase() {
        return passphrase;
    }

    String getInternalAlgorithmName() {
        return internalAlgorithmName;
    }

    byte[] getPassphraseBytes() {
        return passphrase.getBytes(StandardCharsets.UTF_8);
    }
}
